#define DEFINE_GLOBALS
#define TIME
/* #define WRITE_DEC_PICTURE */


#include <sys/time.h>	/* struct timeval on solaris 4 */

#include "pyra/encoder/main.h"

#include "options.h"
#include "config.h"
#include "dispconf.h"
#include "display.h"

#ifndef WITHOUT_CHACODEC
#include "chacodec.h"
#endif

#ifndef WITHOUT_GRABBER
#include "grabber.p"
#endif


static OptionsStruct Init_Options =
{
  NULL,
  USE_VIS,
  USE_DGA,
  USE_SHM,
  DISPLAY,
  0,            /* Default for cover_window: use own toplevel */
  THREAD_YUV,
  GRAY,
  PSEUDOCOLOR,
  EXPAND,
  SIZE_QCIF,
  0,            /* Default for fps */
  LOOP,
  AUDIO
};
 
OptionsStruct Options;


void PrintEncInfo(EncInfo *eInfo) {
  int i;


  /***********************/
  /* File specifications */
  /***********************/
  /* original video sequence */
  printf("\t-o <fname> original sequence [%s]\n", eInfo->orig);
  /* Name of bitstream file */
  printf("\t-b <fname> bitstream basename [%s]\n", eInfo->bitstream);
  printf("\t-bi <num0> <num1> ... <num%d> bitstream layer index\n",
	 NUM_LAYERS);
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->bs_index[i]);
  printf("\n");
  /* Suffix for bitstream files */
  printf("\t-bs <suffix> bitstream suffix [default: no suffix]\n");

  /* Switch display */
  printf("\t-dF turn off the display\n"); 
  /* Format of protocol */
  printf("\t-pF <num> turn on the protocol [%d]\n", eInfo->protFormat);
  printf("\t          0: No protocol\n");
  printf("\t          1: Write protocol (format 1): only total bits and PSNRs\n");
  printf("\t          2: Write protocol (format 2): detailed bits and PSNRs\n");
  printf("\t          3: Write protocol (format 3): coding modes, etc.\n");
  printf("\t          4: Write protocol (format 4): detailed MB, block infos\n");
  /* Name of protocol file */
  printf("\t-p <fname> protocol file name ('-' = stdout) [%s]\n", eInfo->prot);

  /******************************/
  /* Video frame specifications */
  /******************************/
  /* Frame width */
  printf("\t-wF <num> width of frame [%d]\n", eInfo->widthFrame); 
  /* Frame height */
  printf("\t-hF <num> height of frame [%d]\n", eInfo->heightFrame); 
  /* Frame per second of video source */
  printf("\t-fps <num> frames per second of video source [%d]\n", eInfo->fps); 

  /**************************/
  /* Encoder specifications */
  /**************************/
  /* First frame */
  printf("\t-fF <num> first frame [%d]\n", eInfo->firstFrame); 
  /* Last frame */
  printf("\t-lF <num> last frame [%d]\n", eInfo->lastFrame); 
  printf("\t          A negative number means 'loop forever'.\n");
  /* Switch rate control flag */
  printf("\t-rCF turn on the rate control\n");

  /* Set the coding speed */
  printf("\t-cT <num> coding time (speed). 0 = fast; %d is slow but good "
	 "[%d]\n", eInfo->codingTime, eInfo->codingTime);

  /************************/
  /* Layer specifications */
  /************************/
  /* Layer methods */
  printf("\t-m <num0> <num1> ... <num%d> methods used for coding the layers\n",
	 NUM_LAYERS);
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->method[i]);
  printf("\n\t   %d:   Layer is not coded\n"
	 "\t   %d:   Layer is H.263 coded\n"
	 "\t   %d:   Layer is coded with Uwe's Pyramid Coder\n",
	 NONE, H263, PYRA);
  /* Layer scalability types */
  printf("\t-sT <num0> <num1> ... <num%d> scalability types used for the "
	 "layers\n", NUM_LAYERS);
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->scalType[i]);
  printf("\n\t   0:   spatial scalability\n"
	 "\t   1:   temporal scalability\n"
	 "\t   2:   SNR scalability\n");
  /* Layer rates */
  printf("\t-r <num0> <num1> ... <num%d>  rates used for the layers "
	 "[bit/frame]\n", NUM_LAYERS);
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->rate[i]);
  printf("\n\t   Is only used if the rate control flag (rCF) is switch on.\n");
  /* P Layer quantizers */
  printf("\t-qP <num0> <num1> ... <num%d>  quantizers used for the P layers\n",
	 NUM_LAYERS);
  printf("\t    Range: 1...31\n");
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->qP[i]);
  printf("\n\t   Is only used if the rate control flag (rCF) is switch off\n");
  /* I Layer quantizers */
  printf("\t-qI <num0> <num1> ... <num%d>  quantizers used for the I layers\n",
	 NUM_LAYERS);
  printf("\t    Range: 1...31\n");
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->qI[i]);
  printf("\n\t   Is only used if the rate control flag (rCF) is switch off\n");
  /* Layer frame size division factors */
  printf("\t-s <num0> <num1> ... <num%d>  frame size division factors used "
	 "for the layers\n", NUM_LAYERS);
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->sizeDiv[i]);
  printf("\n\t   The numbers give the factors by which the original "
	 "\n\t   frame size is divided\n");
  /* Layer frame frequency division factors */
  printf("\t-f <num0> <num1> ... <num%d>  frequency division factor used for "
	 "the layers\n", NUM_LAYERS);
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->freqDiv[i]);
  printf("\n\t   The numbers give the factors by which the\n"
	 "\t     original frequency is divided\n");
  /* Layer intra frame frequency division factors */
  printf("\t-iF <num0> <num1> ... <num%d>  intra frequency division factor "
	 "used for the layers\n", NUM_LAYERS);
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->intraFreqDiv[i]);
  printf("\n\t   The numbers give the number of frames in this layer from\n"
	 "\t     one intra frame to the next one.\n");
  printf("\t   '0' means that only the first image of the layer is coded in\n"
	 "\t     INTRA mode\n");
  /* Layer start offset */
  printf("\t-sO <num0> <num1> ... <num%d>  start offsets used for the "
	 "layers\n", NUM_LAYERS);
  printf("\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->startOffset[i]);
  printf("\n\t   The numbers give the original frame number in which the "
	 "layer starts.\n");

#ifndef WITHOUT_CHACODEC
  /**********************************************/
  /* Reed Solomon coding and paket multiplexing */
  /**********************************************/
  printf("\n\t-prs switch on the Reed Solomon coding and interleaver packing."
	 " [OFF]\n");
  printf("\t-nIl Number of lines in interleaver [%d].\n", eInfo->nIl);
  printf("\t-bPP Bytes per packet [%d].\n", eInfo->bytesPerPck);
  printf("\t     If == 0: variable from interleaver to interl. (max. %d).\n",
	 MAX_PACKET_SIZE);
  printf("\t-kIl Number of info lines per layer\n\t   ");
  for (i = 0; i < NUM_LAYERS; i++)
    printf("[%d]   ", eInfo->kIl[i]);
  printf("\n");
  printf("\t-pM  Packetization mode [%d]:\n", eInfo->pckMode);
  printf("\t     0: separate layer\n");
  printf("\t     1: iso bytes\n");
  printf("\t     2:  \"    \"  , pack immediately\n");
  printf("\t     3: iso GOBs\n");
  printf("\t     4:  \"   \"  , pack immediately\n");
  printf("\t-pGOP Await a group of pictures (GOB) for packing\n");
  printf("\t      Default: packing at each time instant\n");
  printf("\t-prsFile Filename for packed stream [%s]\n", eInfo->prsFilename);
#endif

  /**************************/
  /* General configurations */
  /**************************/
  /* Switch demo flag */
  printf("\t-demo turn on the demo mode (internal images are displayed\n");
  printf("\t-quiet run quietly (default: verbose)\n");
}


void Usage(char *argv[]) {
  printf("\nScalVico coder (version 1.0)\n");
  printf("Copyright (c) 1997 LNT, Uni Erlangen, Germany\n");
  printf("Type <%s -H> for advanced help\n", argv[0]);
  printf("Usage: %s\n", argv[0]);
}


void VerboseUsage(int argc, char *argv[], EncInfo *eInfo) {
  int i;

  for(i=1;i<argc;++i) {
    /* Help utilities */
    if((!strcmp(argv[i],"-help")) || (!strcmp(argv[i],"-H")) 
       || (!strcmp(argv[i],"-U"))) {
      Usage(argv);
      printf("Options:\n");
      PrintEncInfo(eInfo);
      printf("\n\tDefault options in square brackets are specified in "
	     "main.h\n");
      printf("\n\tFor the Options that need a number for each layer you "
	     "have to specify\n\t*exactly* >%d< numbers!\n", NUM_LAYERS);
      exit(0);
    }
    if((!strcmp(argv[i],"-h")) || (!strcmp(argv[i],"-u"))) {
      Usage(argv);
      exit(0);
    }
  }
}


EncInfo *NewEncInfo()
{
  int         i;
  EncInfo     *eInfo = (EncInfo *)malloc(sizeof(EncInfo));


  if(!eInfo)
    exit(-1);

  /* File specifications */
  eInfo->orig = (char *)malloc(sizeof(char)*MAX_NAME_LENGTH);
  if(!eInfo->orig) exit(-1);
  strcpy(eInfo->orig, DEFAULT_ORIG);

  eInfo->bitstream = (char *)malloc(sizeof(char)*MAX_NAME_LENGTH);
  if(!eInfo->bitstream) exit(-1);
  strcpy(eInfo->bitstream, DEFAULT_BITSTREAM);

  eInfo->bs_suffix = DEFAULT_BS_SUFFIX;

  for (i = 0; i < NUM_LAYERS; i++)
    eInfo->bs_index[i]   = i;

  eInfo->displayFormat   = DEFAULT_DISPLAY_FORMAT;

  eInfo->protFormat  = DEFAULT_PROT_FORMAT;

  eInfo->prot = (char *)malloc(sizeof(char)*MAX_NAME_LENGTH);
  if(!eInfo->prot) exit(-1);
  strcpy(eInfo->prot, DEFAULT_PROT);

  /* Video frame specifications */
  eInfo->widthFrame      = DEFAULT_WIDTH_FRAME;    
  eInfo->heightFrame     = DEFAULT_HEIGHT_FRAME;
  eInfo->fps             = DEFAULT_FPS;

  /* Encoder specifications */
  eInfo->firstFrame      = DEFAULT_FIRST_FRAME;
  eInfo->lastFrame       = DEFAULT_LAST_FRAME;
  eInfo->rateControlFlag = DEFAULT_RATE_CONTROL_FLAG;

  /* Coding speed */
  eInfo->codingTime = DEFAULT_CODING_TIME;

  /* Layer configuration */
  for (i = 0; i < NUM_LAYERS; i++) {
    eInfo->method[i]       = CLIP(DEFAULT_METHOD_0 - i, 0, 2);
    eInfo->rate[i]         = DEFAULT_RATE_0;
    eInfo->qP[i]           = DEFAULT_QUANT_0;
    eInfo->qI[i]           = DEFAULT_QUANT_0;
    eInfo->sizeDiv[i]      = CLIP(DEFAULT_SIZE_DIV_0 << i, 1, 2);
    eInfo->freqDiv[i]      = DEFAULT_FREQ_DIV_0 << i;
    eInfo->intraFreqDiv[i] = DEFAULT_INTRA_FREQ_DIV_0;
    eInfo->startOffset[i]  = DEFAULT_START_OFFSET_0;
    eInfo->scalType[i]     = DEFAULT_SCAL_TYPE;
  }

#ifndef WITHOUT_CHACODEC
  /* Reed Solomon coding and paket multiplexing */
  eInfo->prs = DEFAULT_RS_CODING_FLAG;
  eInfo->nIl = DEFAULT_N_IL;
  eInfo->bytesPerPck = DEFAULT_BYTES_PER_PACKET;
  for (i = 0; i < NUM_LAYERS; i++)
    eInfo->kIl[i] = DEFAULT_K_IL;
  eInfo->pckMode = DEFAULT_PCK_MODE;
  eInfo->awaitGOP = DEFAULT_AWAIT_GOP;

  eInfo->prsFilename = (char *)malloc(sizeof(char)*MAX_NAME_LENGTH);
  if(!eInfo->prsFilename) exit(-1);
  strcpy(eInfo->prsFilename, DEFAULT_PRS_FILENAME);
#endif

  /* General configurations */
  eInfo->demoF = DEFAULT_DEMO_FLAG;
  eInfo->quiet = DEFAULT_QUIET_FLAG;


  return(eInfo);
}


void EvaluateArgumentsEncInfo(int argc, char *argv[], EncInfo *eInfo)
{
  int i, j;
  
  for(i=1;i<argc;++i) {
    /***********************/
    /* File specifications */
    /***********************/
    /* Reads name of file containing original video sequence to be coded */
    if(!strcmp(argv[i],"-o")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      strcpy(eInfo->orig,argv[i]);
      continue;
    }
    /* Name of bitstream file */
    if(!strcmp(argv[i],"-b")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      strcpy(eInfo->bitstream,argv[i]);
      continue;
    }
    if(!strcmp(argv[i],"-bs")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->bs_suffix = (char *)malloc(sizeof(char)*MAX_NAME_LENGTH);
      strcpy(eInfo->bs_suffix,argv[i]);
      continue;
    }
    /* Bitstream indices */
    if(!strcmp(argv[i],"-bi")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->bs_index[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }

    /* Switch display */
    if(!strcmp(argv[i],"-dF")) {
      eInfo->displayFormat = !(eInfo->displayFormat);
      continue;
    }
    /* Format of protocol */
    if(!strcmp(argv[i],"-pF")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->protFormat = atoi(argv[i]);
      continue;
    }
    /* Name of protocol file */
    if(!strcmp(argv[i],"-p")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      strcpy(eInfo->prot,argv[i]);
      continue;
    }

    /******************************/
    /* Video frame specifications */
    /******************************/
    /* Frame width */
    if(!strcmp(argv[i],"-wF")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->widthFrame = atoi(argv[i]);
      continue;
    }
    /* Frame height */
    if(!strcmp(argv[i],"-hF")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->heightFrame = atoi(argv[i]);
      continue;
    }
    /* Frames per second of video source */
    if(!strcmp(argv[i],"-fps")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->fps = atoi(argv[i]);
      continue;
    }

    /**************************/
    /* Encoder specifications */
    /**************************/
    /* First frame */
    if(!strcmp(argv[i],"-fF")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->firstFrame = atoi(argv[i]);
      continue;
    }
    /* Last frame */
    if(!strcmp(argv[i],"-lF")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->lastFrame = atoi(argv[i]);
      continue;
    }
    /* Switch rate control flag */
    if(!strcmp(argv[i],"-rCF")) {
      eInfo->rateControlFlag = !(eInfo->rateControlFlag);
      continue;
    }

    /* Set coding time (speed) */
    if(!strcmp(argv[i],"-cT")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->codingTime = atoi(argv[i]);
      continue;
    }

    /************************/
    /* Layer specifications */
    /************************/
    /* Layer methods */
    if(!strcmp(argv[i],"-m")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->method[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Layer scalability types */
    if(!strcmp(argv[i],"-sT")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->scalType[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Layer rates */
    if(!strcmp(argv[i],"-r")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->rate[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Layer quantizers */
    if(!strcmp(argv[i],"-qP")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->qP[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Layer quantizers */
    if(!strcmp(argv[i],"-qI")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->qI[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Layer frame size division factors */
    if(!strcmp(argv[i],"-s")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->sizeDiv[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Layer frame frequency division factors */
    if(!strcmp(argv[i],"-f")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->freqDiv[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Layer intra frame frequency division factors */
    if(!strcmp(argv[i],"-iF")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->intraFreqDiv[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Layer start offset */
    if(!strcmp(argv[i],"-sO")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->startOffset[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }

#ifndef WITHOUT_CHACODEC
    /**********************************************/
    /* Reed Solomon coding and paket multiplexing */
    /**********************************************/
    /* Flag to switch on Reed Solomon coding */
    /* Switch rate control flag */
    if(!strcmp(argv[i],"-prs")) {
      eInfo->prs = !(eInfo->prs);
      continue;
    }
    /* Number of lines in interleaver */
    if(!strcmp(argv[i],"-nIl")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->nIl = atoi(argv[i]);
      continue;
    }
    /* Bytes per packet */
    if(!strcmp(argv[i],"-bPP")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->bytesPerPck = atoi(argv[i]);
      continue;
    }
    /* Number of info lines per layer */
    if(!strcmp(argv[i],"-kIl")) {
      if(i+1 >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      for (j = 0; j < NUM_LAYERS; j++) {
	if (++i >= argc || *argv[i] == '-') {
	  --i;
	  break;
	}
	eInfo->kIl[j] = atoi(argv[i]);
      }
      if (j == 0) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      continue;
    }
    /* Packetization mode */
    if(!strcmp(argv[i],"-pM")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      eInfo->pckMode = atoi(argv[i]);
      continue;
    }
    /* Await GOP for packeing */
    if(!strcmp(argv[i],"-pGOP")) {
      eInfo->awaitGOP = !(eInfo->awaitGOP);
      continue;
    }
    /* Name of bitstream file */
    if(!strcmp(argv[i],"-prsFile")) {
      if(++i >= argc) {Usage(argv); printf("%d %d\n",i,argc); exit(0);}
      strcpy(eInfo->prsFilename,argv[i]);
      continue;
    }
#endif

    /**************************/
    /* General configurations */
    /**************************/
    /* Switch demo flag */
    if(!strcmp(argv[i],"-demo")) {
      eInfo->demoF = !(eInfo->demoF);
      continue;
    }
    /* Switch quiet flag */
    if(!strcmp(argv[i],"-quiet")) {
      eInfo->quiet = 1;
      continue;
    }
    printf("Unknown option '%s'\n", argv[i]);
    PrintEncInfo(eInfo);
    exit(1);
    break;
  }


  /* Set protocol to stdout if the filename is '-' */
  if (strcmp(eInfo->prot, "-") == 0) {
    eInfo->protFile = stdout;
  } else {
    if ((eInfo->protFile = fopen(eInfo->prot, "w")) == NULL) {
      fprintf(stderr, "Can not open protocol file >%s<\n", eInfo->prot);
      exit (0);
    }
  }
#ifdef PROTOCOL
  protFormat_g = eInfo->protFormat;
  if (eInfo->protFormat > 0) {
    fp_prot_g = eInfo->protFile;
    for (i = 0; i < argc; ++i)
      fprintf(fp_prot_g, "%s ", argv[i]);
    fprintf(fp_prot_g, "\n\n");
  } else
    fp_prot_g = NULL;
#endif
}







/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PCoder -- Scalable pyramid coder (ScalVico)
 *
 * Author:      K.S.
 *	        
 * Created:     25-Aug-97
 *	        
 * Purpose:     -
 * 	        
 * Options:     -o <fname> original sequence 
 *		           [/net/nt36/cdrom/unnamed_cdrom/foreman/fm]
 *	        -b <fname> bitstream [bitstream]
 *	        -bs<suffix> suffix of bitstream (default: NULL = no suffix)
 *              -bi <num0> <num1> ... <num%d> bitstream layer index
 *                  [0] [1] [2] [3]
 *
 *              -dF turn off the display
 *	        -pF protocol switch
 *	        -p <fname> protocol [prot]
 *
 *	        -wF <num> width of frame [352]
 *	        -hF <num> height of frame [288]
 *              -fps <num> frames per second of video source [30]
 *
 *	        -fF <num> first frame [0]
 *	        -lF <num> last frame [100]
 *                        A negative number means 'loop forever'.
 *              -rCF turn on the rate control
 *
 *              -cT <num> coding time (speed). 0 = fast; %d is slow but good
 *
 *              -m  <num> <num> <num> <num> Layer coding methods
 *                  [2] [1] [0] [0]
 *                  0:   NONE
 *                  1:   H263
 *                  2:   PYRA
 *              -sT <num> <num> <num> <num> Scalability type (only Pyra layers)
 *                  0:   spatial (and temporal) scalability
 *                  1:   temporal scalability
 *                  2:   SNR scalability
 *              -r  <num> <num> <num> <num> Layer rates (if rCF is on)
 *                  [-1] [-1] [-1] [-1]
 *              -q  <num> <num> <num> <num> Layer quantizers
 *                  [10] [10] [10] [10]
 *              -s  <num> <num> <num> <num> Layer frame size division factors
 *                  [1] [2] [4] [8]
 *              -f  <num> <num> <num> <num> Layer frame frequency div. factors
 *                  [1] [2] [4] [8]
 *              -demoF turn on demo mode
 *              -quiet run quietly
 *
 *	        Default options in square brackets are specified in main.h
 *
 *
 * Description: -
 *
 * See also:    -
 *
 * Modified:    -
 *
 *****************************************************************************/
void main(int argc, char *argv[])
/***********************************************************CommentEnd********/
{
#ifdef TIME
  int            runtime;
  struct timeval tstart, tstop;
#endif
  EncInfo        *eInfo = NewEncInfo();

#ifdef WRITE_DEC_PICTURE
  char           decPictName[100];
#endif
#ifdef UWEGDB
  char           name[80];
#endif
  int            grab_f;
  int            l, ll, numLayer;
  int            n, n0;
  int            lastCodedLayer, baseLayer, baseLayerFlag;
  int            topLayer;
  int            countH263 = 0;
  int            nBits = 0;
  double         psnr[3], totalPSNR[NUM_LAYERS][3];
  int            percentIMBs = 0;
  int            q;
  int            w, h, wSub, hSub;
  int            dSizeAcc = 0, dSize[NUM_LAYERS];
  int            pictureCodingType[NUM_LAYERS],
                 prevPictureCodingType[NUM_LAYERS], gfid[NUM_LAYERS];
  int            totalFrames[NUM_LAYERS];
  unsigned long  totalBits[NUM_LAYERS];
  char           *bitstrName[NUM_LAYERS];
  Bitstr         *bitstr[NUM_LAYERS];
  MVField        mvFieldIn[NUM_LAYERS], mvFieldOut[NUM_LAYERS];
  int            *mbInd[NUM_LAYERS];
  Picture        *pict, *origPict[NUM_LAYERS], *prevOrigPict[NUM_LAYERS],
                 *decPict[NUM_LAYERS],
                 *spatPredPict[NUM_LAYERS], *tempPredPict[NUM_LAYERS];
  Picture        *dispPict;
  Byte           *interpol[NUM_LAYERS][3];
#ifndef WITHOUT_CHACODEC
  int            varBytesPerPck_f = 0;
  int            bytesPerPck, maxBytesPerPck = 0;
  int            bytesPerLayerPck[NUM_LAYERS];
  int            nMBsPerGOB[NUM_LAYERS];
  int            *gobSize[NUM_LAYERS];
  int            nGOBsPerGOP[NUM_LAYERS];
  int            nGOBsPerFrame[NUM_LAYERS];
  int            nGOBsCoded[NUM_LAYERS];
  int            maxFreqDiv = 1;
  int            forceLayer[NUM_LAYERS];
  int            pck2Write;
  Byte           *packStream = NULL;
  FILE           *fp_prs = NULL;
#endif


  /* Evalute command line arguments and/or print usage */
  EvaluateArgumentsEncInfo(argc, argv, eInfo);
  VerboseUsage(argc, argv, eInfo);

  grab_f = (eInfo->orig[0] == '-');
  /* Print a short hello message */
  if (!eInfo->quiet) {
    printf("\n\n                  ScalVico\n\n");
    printf("    Uwe Horn, Jrgen Weigert, Klaus Stuhlmller\n\n\n");
  }

  Options = Init_Options;       /* struct copy */

  /* How many layer do we have to code? */
  for (numLayer = 0; numLayer < NUM_LAYERS; numLayer++)
    if (eInfo->method[numLayer] == H263) {
      countH263++;
      if (countH263 > 1) {
	fprintf(stderr, "Only one layer can be coded with H.263!\n");
	exit(0);
      }
    } else if (eInfo->method[numLayer] == NONE)
      break;

#ifndef WITHOUT_CHACODEC
  /************************/
  /* Init channel encoder */
  /************************/
  if (eInfo->prs) {
    GFInit(8);   /* 8 == bits per byte */
    for (l = 0; l < numLayer; l++) {
      if (eInfo->kIl[l] > eInfo->nIl || eInfo->kIl[l] < K_IL_INFO) {
	fprintf(stderr, "'kIl[.]' must be <= 'nIl' and >= %d for each layer\n",
		K_IL_INFO);
	exit(1);
      }
    }
    if (eInfo->bytesPerPck <= 0) {
      varBytesPerPck_f = 1;
      eInfo->bytesPerPck = MAX_PACKET_SIZE;
    }
    bytesPerPck = eInfo->bytesPerPck;
    maxBytesPerPck = bytesPerPck;
    packStream = (Byte *)malloc(bytesPerPck * eInfo->nIl * sizeof(Byte));
    if (eInfo->pckMode >= 3) {
      for (l = 0; l < numLayer; l++)
	maxFreqDiv = MAX(maxFreqDiv, eInfo->freqDiv[l]);
      for (l = 0; l < numLayer; l++) {
	nMBsPerGOB[l] = (eInfo->widthFrame / eInfo->sizeDiv[l]) / 16;
	nGOBsPerFrame[l] = (eInfo->heightFrame / eInfo->sizeDiv[l]) / 16;
	nGOBsPerGOP[l] = (maxFreqDiv / eInfo->freqDiv[l]) * nGOBsPerFrame[l];
	nGOBsCoded[l] = 0;
	mbInd[l] = (int *)malloc(nGOBsPerFrame[l] * nMBsPerGOB[l]*sizeof(int));
	gobSize[l] = (int *)malloc(nGOBsPerGOP[l] * sizeof(int));
      }
    } else {
      for (l = 0; l < numLayer; l++) {
	mbInd[l] = NULL;
	gobSize[l] = NULL;
      }
    }
    fp_prs = OpenInitPrsFile(eInfo->prsFilename, bytesPerPck, eInfo->nIl,
			     eInfo->pckMode, numLayer, eInfo->method);
  } else
#endif
    for (l = 0; l < numLayer; l++)
      mbInd[l] = NULL;



  if (!eInfo->quiet)
    printf("Allocating memory\n");
  /* Allocate memory */
  pict = AllocPicture(eInfo->widthFrame, eInfo->heightFrame);
  dispPict = ClonePicture(pict);

#ifdef COUNT_BITS
  bitCount = (struct bit_count_struct *)calloc(NUM_LAYERS,sizeof(struct bit_count_struct));
#endif

  /* Alloc and init layer */
  for (l = 0; l < numLayer; l++) {

    /* Init counter */
    totalBits[l] = 0;
    totalPSNR[l][0] = 0.0;
    totalPSNR[l][1] = 0.0;
    totalPSNR[l][2] = 0.0;
    totalFrames[l] = 0;

    prevPictureCodingType[l] = -1;
    gfid[l] = -1;

    /* Check arguments */
    ASSERT(eInfo->qP[l] > 0 && eInfo->qP[l] < 32 &&
	   eInfo->qI[l] > 0 && eInfo->qI[l] < 32 &&
	   eInfo->freqDiv[l] > 0 &
	   eInfo->intraFreqDiv[l] >= 0 && eInfo->startOffset[l] >= 0);
    switch(eInfo->sizeDiv[l]) {
    case 1:  dSize[l] = 0; break;
    case 2:  dSize[l] = 1; break;
    case 4:  dSize[l] = 2; break;
    case 8:  dSize[l] = 3; break;
    case 16: dSize[l] = 4; break;
    case 32: dSize[l] = 5; break;
    case 64: dSize[l] = 6; break;
    default:
      fprintf(stderr, "Only 1,2,4,8,16,32 and 64 are allowed size division "
	      "factors! (eInfo->sizeDiv[l] is %d)\n",eInfo->sizeDiv[l]);
      exit(1);
    }
    if (l > 0) {
      dSize[l] -= dSizeAcc;
    }
    dSizeAcc += dSize[l];
    ASSERT(dSize[l] >= 0);

#ifndef WITHOUT_CHACODEC
    if (eInfo->prs) {
      ASSERT(bitstr[l] = AllocBitstr(2 * 8 * eInfo->widthFrame *
				     eInfo->heightFrame));
    } else
#endif
    {
      /* Open bitstreams */
      if (eInfo->bs_suffix == NULL) {
	bitstrName[l] = (char *)malloc((strlen(eInfo->bitstream) + 10)
				       * sizeof(char));
	sprintf(bitstrName[l], "%s%d", eInfo->bitstream, eInfo->bs_index[l]);
      } else {
	bitstrName[l] = (char *)malloc((strlen(eInfo->bitstream) + 10 +
					strlen(eInfo->bs_suffix))
				       * sizeof(char));
	sprintf(bitstrName[l], "%s%d.%s", eInfo->bitstream, eInfo->bs_index[l],
		eInfo->bs_suffix);
      }
      ASSERT(bitstr[l] = FopenBitstr(2 * 8 * eInfo->widthFrame *
				     eInfo->heightFrame, bitstrName[l], "w"));
      if (!eInfo->quiet)
	printf("Creating bitstream '%s'\n", bitstrName[l]);
    }

    /* Original image for the layer */
    w = eInfo->widthFrame / eInfo->sizeDiv[l];
    h = eInfo->heightFrame / eInfo->sizeDiv[l];
    wSub = w;
    hSub = h;
    if (w % MACROBLOCK_SIZE)
      w = ((w / MACROBLOCK_SIZE) + 1) * MACROBLOCK_SIZE;
    if (h % MACROBLOCK_SIZE)
      h = ((h / MACROBLOCK_SIZE) + 1) * MACROBLOCK_SIZE;
    origPict[l] = AllocResetPicture(w, h);
    origPict[l]->ws = wSub;
    origPict[l]->hs = hSub;
    /* Previous original layer image */
    prevOrigPict[l] = ClonePicture(origPict[l]);
    /* Decoded/Reconstructed layer image */
    decPict[l] = CloneResetPicture(origPict[l]);
    /* Temporal prediction reference layer image */
    tempPredPict[l] = CloneResetPicture(origPict[l]);
    /* Interpolated versions of tempPredPict */
    interpol[l][0] = (Byte *)malloc(tempPredPict[l]->w * tempPredPict[l]->h *
				    sizeof(Byte));
    interpol[l][1] = (Byte *)malloc(tempPredPict[l]->w * tempPredPict[l]->h *
				    sizeof(Byte));
    interpol[l][2] = (Byte *)malloc(tempPredPict[l]->w * tempPredPict[l]->h *
				    sizeof(Byte));

    /* Output Motion vector fields */
    mvFieldOut[l].w = origPict[l]->w / 16;
    mvFieldOut[l].h = origPict[l]->h / 16;
    mvFieldOut[l].mode = malloc(mvFieldOut[l].w	* mvFieldOut[l].h 
				* sizeof(short));
    mvFieldOut[l].mx = malloc(mvFieldOut[l].w * mvFieldOut[l].h
			      * sizeof(short));
    mvFieldOut[l].my = malloc(mvFieldOut[l].w * mvFieldOut[l].h
			      * sizeof(short));

    if (l < numLayer - 1) {
      /* spatial reference image */
      spatPredPict[l] = CloneResetPicture(origPict[l]);
      /* Input Motion vector fields */
      mvFieldIn[l].w = mvFieldOut[l].w;
      mvFieldIn[l].h = mvFieldOut[l].h;
      mvFieldIn[l].mode = malloc(mvFieldIn[l].w * mvFieldIn[l].h
				 * sizeof(short));
      mvFieldIn[l].mx = malloc(mvFieldIn[l].w * mvFieldIn[l].h
			       * sizeof(short));
      mvFieldIn[l].my = malloc(mvFieldIn[l].w * mvFieldIn[l].h
			       * sizeof(short));
    }
  }

  /******************/
  /* Init Uwes code */
  /******************/
  InitFilters(eInfo->widthFrame/2, eInfo->heightFrame/2);
  InitBlockPyra();
  InitExtVectTables();
  InitHuffTables();
  InitLatticeQuantizer();
  if (eInfo->displayFormat == ON) {
    init_display(origPict[0]->ws, origPict[0]->hs);
  }

  if (!eInfo->quiet)
    printf("And here we go ...\n");

#ifdef TIME
  gettimeofday(&tstart,(struct timezone *)NULL);
#endif


  /******************************************************************/
  /******************************************************************/
  /**                          Video loop                          **/
  /******************************************************************/
  /******************************************************************/
  for (n = eInfo->firstFrame, n0 = 0;
       (n <= eInfo->lastFrame || eInfo->lastFrame < 0); n++, n0++) {
    if (!eInfo->quiet) {         /* set breakpoint UWE */
      printf("%4d", n);
      fflush(stdout);
    }
    if (grab_f) {
#ifndef WITHOUT_GRABBER
      grabPicture(pict);
#else
      fprintf(stderr, "Sorry, Encoder compiled WITHOUT_GRABBER support\n");
      exit(0);
#endif
    } else {
      if (!ReadPicture(eInfo->orig, SEP_RAW_TYPE, n, pict)) {
	fprintf(stderr, "Can't find input sequence \"%s\"\n", eInfo->orig);
	exit(-1);
      }
    }

    /****************/
    /* Downsampling */
    /****************/
    /* Check which layer will be coded this time */
    for (l = 0, baseLayer = 0; l < numLayer; l++)
      /* Check the layer frame frequency */
      if ((n0 - eInfo->startOffset[l]) >= 0 &&
	  (n0 - eInfo->startOffset[l]) % eInfo->freqDiv[l] == 0)
	baseLayer = l;
    /* Downsampling of original image */
    MultDownsamplePictureSub(dSize[0], pict, origPict[0]);
    for (l = 1; l <= baseLayer; l++)
      MultDownsamplePictureSub(dSize[l], origPict[l - 1], origPict[l]);
    /***********************/
    /* End of Downsampling */
    /***********************/

    /**************/
    /* Layer loop */
    /**************/
    lastCodedLayer = -1;
    topLayer = -1;
    for (l = numLayer - 1; l >= 0; l--) {
      /* Check the layer frame frequency */
      if ((n0 - eInfo->startOffset[l]) >= 0 &&
	  (n0 - eInfo->startOffset[l]) % eInfo->freqDiv[l] == 0) {

	if (!eInfo->quiet) {
	  printf("      L = %d", l);
	  fflush(stdout);
	}

	topLayer = l;


	/* Update temporal reference images */
	SwapPictures(&tempPredPict[l], &decPict[l]);

#ifdef COUNT_BITS
	bitCount[l].totalFrames++;
	bitCountP = bitCount + l;
#endif

	/*************************/
	/* Decide on coding mode */
	/*************************/
	/* Set coding mode INTRA or INTER */
	if (eInfo->intraFreqDiv[l] == 0) {
	  /* If the intra frequency division factor is 0,
	     only the first layer image is coded in INTRA mode
	     (except a temporal scalability layer) */
	  if ((n0 - eInfo->startOffset[l]) == 0 && eInfo->scalType[l] != 2) {
	    pictureCodingType[l] = PICTURE_CODING_TYPE_INTRA;
	    q = eInfo->qI[l];
	  } else {
	    pictureCodingType[l] = PICTURE_CODING_TYPE_INTER;
	    q = eInfo->qP[l];
	  }
	} else {
	  if (((n0 - eInfo->startOffset[l]) / eInfo->freqDiv[l])
	      % eInfo->intraFreqDiv[l] == 0) {
	    pictureCodingType[l] = PICTURE_CODING_TYPE_INTRA;
	    q = eInfo->qI[l];
	  } else {
	    pictureCodingType[l] = PICTURE_CODING_TYPE_INTER;
	    q = eInfo->qP[l];
	  }
	}
	if (prevPictureCodingType[l] != pictureCodingType[l])
	  gfid[l]++;
	prevPictureCodingType[l] = pictureCodingType[l];

	if (eInfo->method[l] == PYRA &&
	    pictureCodingType[l] == PICTURE_CODING_TYPE_INTER) {
	  /* Init halfpel interpolation images */
	  InitHalfPelInterpolation(tempPredPict[l], interpol[l]);
	}

	baseLayerFlag = (l == baseLayer);

	if (!baseLayerFlag && eInfo->method[l] != H263) {
	  /* Prepare information from lower layers */
	  if (lastCodedLayer > 0 &&
	      (pictureCodingType[l] == PICTURE_CODING_TYPE_INTER)) {
	    /* Upsampling of motion vector field */
	    SampleUpMVField(mvFieldOut[lastCodedLayer],
			    eInfo->freqDiv[lastCodedLayer],
			    mvFieldIn[l], eInfo->freqDiv[l]);
	  } else {
	    lastCodedLayer = l + 1;
	  }

	  /* Upsampling of reconstructed image */
	  dSizeAcc = 0;
	  for (ll = lastCodedLayer; ll > l; ll--)
	    dSizeAcc += dSize[ll];
	  MultInterpolateFastPictureSub(dSizeAcc, decPict[lastCodedLayer],
					spatPredPict[l]);
	}

	lastCodedLayer = l;

	if (eInfo->protFormat > 0) {
	  if (eInfo->protFormat > 1)
	    fprintf(eInfo->protFile,"\n###################################\n");
	  fprintf(eInfo->protFile, "frame: %d layer: %d Coding mode: %s ",
		  n, l, pictureCodingType[l] ? "P" : "I");
	  if (eInfo->protFormat > 1)
	    fprintf(eInfo->protFile,"\n###################################\n");
	}

	/**************/
	/* Code layer */
	/**************/
	if (eInfo->rateControlFlag) {
	  /* Code at the given rate */
	  if (eInfo->method[l] == H263) {
	    nBits = EncodeH263(eInfo->rate[l], eInfo->codingTime,
			       pictureCodingType[l], n0, percentIMBs,
			       origPict[l], prevOrigPict[l],
			       tempPredPict[l], NULL, decPict[l],
			       bitstr[l], mbInd[l], NULL, mvFieldOut + l);
	  } else if (eInfo->method[l] == PYRA) {
	    nBits = CodePyraLayer(origPict[l], prevOrigPict[l],
				  spatPredPict[l], tempPredPict[l],
				  interpol[l], mvFieldIn[l], eInfo->rate[l],
				  pictureCodingType[l], eInfo->scalType[l],
				  gfid[l], n0, bitstr[l], decPict[l],
				  mvFieldOut[l], mbInd[l], baseLayerFlag,
				  eInfo->codingTime);
	  }
	} else {
	  /* Code with the given quantizer or scale */
#ifdef UWEGDB
            sprintf(name,"orig-%d.%d.%d", pictureCodingType[l], l, n);
            WritePGMFile(name, origPict[l]->w, origPict[l]->h, origPict[l]->y);
 
            sprintf(name,"spinter-%d.%d.%d", pictureCodingType[l], l, n);
            WritePGMFile(name, spatPredPict[l]->w, spatPredPict[l]->h, spatPredP
ict[l]->y);
 
            sprintf(name,"tpinter-%d.%d.%d", pictureCodingType[l], l, n);
            WritePGMFile(name, tempPredPict[l]->w, tempPredPict[l]->h, tempPredP
ict[l]->y);
#endif
	  if (eInfo->method[l] == H263) {
	    nBits = EncodeH263Q(q, eInfo->codingTime, pictureCodingType[l], n0,
				percentIMBs, origPict[l], prevOrigPict[l],
				tempPredPict[l],
				NULL, decPict[l], bitstr[l], mbInd[l], NULL,
				mvFieldOut + l);
	  } else if (eInfo->method[l] == PYRA) {
	    nBits = CodePyraLayerQ(NULL, origPict[l], prevOrigPict[l],
				   spatPredPict[l], tempPredPict[l],
				   interpol[l], mvFieldIn[l],
				   q, pictureCodingType[l],
				   eInfo->scalType[l], gfid[l], n0,
				   bitstr[l], decPict[l],
				   mvFieldOut[l], mbInd[l], baseLayerFlag,
				   eInfo->codingTime);
	  }
	}

	if (!eInfo->quiet || eInfo->protFormat > 0) {
	  psnr[0] = PSNRSubPictures(origPict[l]->y, decPict[l]->y,
				    origPict[l]->ws, origPict[l]->hs,
				    origPict[l]->w);
	  psnr[1] = PSNRSubPictures(origPict[l]->u, decPict[l]->u,
				    origPict[l]->ws/2, origPict[l]->hs/2,
				    origPict[l]->w/2);
	  psnr[2] = PSNRSubPictures(origPict[l]->v, decPict[l]->v,
				    origPict[l]->ws/2, origPict[l]->hs/2,
				    origPict[l]->w/2);
	}

#ifdef PROTOCOL
	if (eInfo->protFormat > 0) {
	  if (eInfo->protFormat >= 3)
	    fprintf(eInfo->protFile, "\n\n");
	  fprintf(eInfo->protFile, "Rate: %6d     PSNR: %2.2f dB "
		  "( %2.2f / %2.2f )\n\n", nBits, psnr[0], psnr[1], psnr[2]);
	}
#endif
	/* Some information to stdout */
	if (!eInfo->quiet) {
	  printf("        Rate: %6d     PSNR: %2.2fdB (%2.2f/%2.2f)"
		 "\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b"
		 "\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b",
		 nBits, psnr[0], psnr[1], psnr[2]);
	  fflush(stdout);
	}

	if (nBits < 0) {
	  fprintf(stderr, "Coding error!\n");
	  exit (-1);
	}

	totalBits[l] += nBits;
	totalPSNR[l][0] += psnr[0];
	totalPSNR[l][1] += psnr[1];
	totalPSNR[l][2] += psnr[2];
	totalFrames[l]++;

	if (bitstr[l]->ind >= bitstr[l]->size) {
	  fprintf(stderr, "Alarm: Allocated bitstream is too small!\n");
	  fprintf(stderr, "Allocate %d bits instead of %d.\n", bitstr[l]->ind, bitstr[l]->size);
	  exit(1);
	}

#ifndef WITHOUT_CHACODEC
	forceLayer[l] = 0;
	if (eInfo->prs) {
	  if (eInfo->pckMode >= 3) {
	    mbInd2gobSize(mbInd[l], bitstr[l]->ind, nMBsPerGOB[l],
			  nGOBsPerFrame[l], gobSize[l] + nGOBsCoded[l]);
	    nGOBsCoded[l] += nGOBsPerFrame[l];
	  }
	} else
#endif
	  CheckOutBuffer(bitstr[l]);

#ifdef WRITE_DEC_PICTURE
	/* Write decoded picture */
	if (n0 == 0) {
	  sprintf(decPictName, "rm decPict%d", l);
	  system(decPictName);
	}
	sprintf(decPictName, "decPict%d", l);
	WritePicture(decPictName, CAT_SEQ_TYPE, n, decPict[l]);
#endif

	/* Update previous original images */
	SwapPictures(&prevOrigPict[l], &origPict[l]);

	if (eInfo->demoF) {
	  printf("\nDecoded Image\t\t");
	  printf("Press Return: ");
	  getchar();
	}

	if (!eInfo->quiet)
	  printf("\b\b\b\b\b\b\b\b\b\b\b\b");
      }
#ifndef WITHOUT_CHACODEC
      else {
	forceLayer[l] = 1;
      }
#endif
    }
    /*********************/
    /* End of Layer loop */
    /*********************/
    if (!eInfo->quiet)
      printf("\r");
    /* Display */
    if (eInfo->displayFormat == ON && topLayer >= 0) {
      yuv2rgb(decPict[topLayer]);
    }

#ifndef WITHOUT_CHACODEC
    if (eInfo->prs) {
      if (!eInfo->awaitGOP || (n0 + 1) % maxFreqDiv == 0) {
	/* Channel coding; Paketizing */
	while ((pck2Write = chaenco(numLayer, gobSize, nGOBsCoded, eInfo->kIl,
				    eInfo->nIl, eInfo->pckMode,
				    varBytesPerPck_f,
				    maxBytesPerPck, &bytesPerPck,
				    bytesPerLayerPck, forceLayer,
				    bitstr, packStream, 0)) > 0) {
	  /* Write packBitstr to file/port */
	  writeInterleaverRTP(packStream, bytesPerPck, pck2Write, fp_prs);
	  for (l = 0; l < numLayer; l++)
	    nGOBsCoded[l] = 0;
	}
      }
    } else
#endif
      {
	/*## hier wird buffer geleert. Alles weg !! ## */
	for (l=0;l<numLayer;l++)
	  CheckOutBuffer(bitstr[l]);
      }
  }
  /*********************/
  /* End of Video loop */
  /*********************/


#ifndef WITHOUT_CHACODEC
  if (eInfo->prs) {
    /* Channel coding; Paketizing */
    while ((pck2Write = chaenco(numLayer, gobSize, nGOBsCoded, eInfo->kIl,
				eInfo->nIl, eInfo->pckMode,
				varBytesPerPck_f, maxBytesPerPck,
				&bytesPerPck, bytesPerLayerPck, forceLayer,
				bitstr, packStream,1)) > 0) {
      /* Write packBitstr to file/port */
      writeInterleaverRTP(packStream, bytesPerPck, pck2Write, fp_prs);
    }
    /* Close packet stream file */
    fclose(fp_prs);
    /* Clean up memory */
    if (eInfo->pckMode >= 3) {
      for (l = 0; l < numLayer; l++) {
	free(mbInd[l]);
	free(gobSize[l]);
      }
    }
    GFFree();
  }
#endif


  if (!eInfo->quiet) {
    /* Print runtime */
#ifdef TIME
    gettimeofday(&tstop, (struct timezone *)NULL);
    runtime = 100 * (tstop.tv_sec - tstart.tv_sec)
      + (tstop.tv_usec - tstart.tv_usec) / 10000;
    printf("\n\n%d.%02d seconds, %d frames, %f fps (realtime: %d fps)\n",
	   runtime / 100, runtime % 100, n0,
	   (float)n0 / (float)runtime * 100.0, eInfo->fps);
#endif
    /* Print some statistics */
    printf("\n");
    for (l = 0; l < numLayer; l++)
      if (totalFrames[l] > 0)
	printf("Layer %d:   %3d frames coded with %8d bits / %2.2f dB PSNR "
	       "(%2.2f/%2.2f).\t%6d bpf\t%7d bps\n",
	       l, totalFrames[l], (int)(totalBits[l]),
	       totalPSNR[l][0] / (double)totalFrames[l],
	       totalPSNR[l][1] / (double)totalFrames[l],
	       totalPSNR[l][2] / (double)totalFrames[l],
	       (int)(totalBits[l] / totalFrames[l]),
	       (int)(eInfo->fps * totalBits[l] /n0));
      else
	printf("Layer %d:   no frames coded.\n", l);
  }

  if (eInfo->protFormat > 0) {
    /* Print some statistics */
    printf("\n");
    for (l = 0; l < numLayer; l++)
      if (totalFrames[l] > 0)
	fprintf(eInfo->protFile, "Layer %d:   %3d frames coded with %8d bits / "
		"%2.2f dB PSNR ( %2.2f / %2.2f ).\t%6d bpf\t%7d bps\n",
		l, totalFrames[l], (int)(totalBits[l]),
		totalPSNR[l][0] / (double)totalFrames[l],
		totalPSNR[l][1] / (double)totalFrames[l],
		totalPSNR[l][2] / (double)totalFrames[l],
		(int)(totalBits[l] / totalFrames[l]),
		(int)(eInfo->fps * totalBits[l] /n0));
      else
	fprintf(eInfo->protFile, "Layer %d:   no frames coded.\n", l);
  }

#ifdef COUNT_BITS
  for (l = 0; l < numLayer; l++)
    printf("%d  %d  %d  %d  %d  %d  %d  %d  %d\n", bitCount[l].totalFrames,
	   bitCount[l].totalBits, bitCount[l].headerBits,
	   bitCount[l].motionBits, bitCount[l].lumBits, bitCount[l].lum1Bits,
	   bitCount[l].lum2Bits, bitCount[l].lum3Bits, bitCount[l].chromBits);
  free(bitCount);
#endif

  /* Clean up memory */
  DisinitFilters();
  FreePicture(pict);
  FreePicture(dispPict);
  for (l = 0; l < numLayer; l++) {
#ifndef WITHOUT_CHACODEC
    if (eInfo->prs) {
      FreeBitstr(bitstr + l);
    } else
#endif
    {
      free(bitstrName[l]);
      FcloseBitstr(bitstr + l);
    }
    FreePicture(origPict[l]);
    FreePicture(prevOrigPict[l]);
    FreePicture(decPict[l]);
    FreePicture(tempPredPict[l]);
    free(interpol[l][0]);
    free(interpol[l][1]);
    free(interpol[l][2]);
    free(mvFieldOut[l].mode);
    free(mvFieldOut[l].mx);
    free(mvFieldOut[l].my);
    if (l < numLayer - 1) {
      FreePicture(spatPredPict[l]);
      free(mvFieldIn[l].mode);
      free(mvFieldIn[l].mx);
      free(mvFieldIn[l].my);
    }
  }
  sleep(1);
  if (eInfo->displayFormat == ON) {
    exit_display();
  }

#ifdef TRAINING
  PrintHuffCodeHistos();
#endif

  if (!eInfo->quiet) {
    printf("\n\t***************");
    printf("\n\t*   The End   *");
    printf("\n\t***************\n");
  }
}
