#include <limits.h>

#include "defs.h"
#include "Util.h"
#undef ABS
#define ABS labs
#include "structs.h"
#include "motion.h"
#include "bitOut.h"
#include "pyraencoder.h"

#include "Lattice.h"
#include "pyracoder.p"
#include "common.h"
#include "h263/encoder/h263encoder.h"
#include "h263encoder.p"


#define ISCALE0(x) (x)
#define ISCALE1(x) ((x)/2)


#define RD_MODE_DECISION

/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CodePyraLayer -- Code one layer of the pyramid with VQ
 *
 * Author:           K.S.
 *
 * Created:          1-Sep-97
 *
 * Purpose:          One layer of the pyramid is coded as described in Uwe's
 *                   Dis.
 *
 * Arguments in:     Picture    *origPict       original image
 *                                              (possibly downsampled)
 *                   Picture    *prevOrigPict   previous original image
 *                   Picture    *spatPredPict   spatial prediction image
 *                                              (decoded image from
 *                                              lower layer).
 *                   Picture    *tempPredPict   temporal prediction image
 *                                              (last decoded image from this
 *                                               layer).
 *                   Byte       *interpol[3]    Halfpel interpolated versions
 *                                              of 'tempPredPict'.
 *                   MVField    mvIn            Input motion vector field
 *                                              (MVField from lower layer).
 *                   int        rate            target bit rate for this
 *                                              picture.
 *                   int        codingType      PICTURE_CODING_TYPE_INTRA or
 *                                              PICTURE_CODING_TYPE_INTER.
 *                   int        scalType        scalability type.
 *                                               0: spatial (and temp.) scal.
 *                                               1: temporal scalability
 *                                               1: SNR scalability
 *                   int        t               time reference to be put into
 *                                              the bitstream.
 *                   int        baseLayer_f     1 is this is the base layer,
 *                                              0 otherwise.
 *                   int        codingTime      0... select between fast and
 *                                              and slow/good coding algorithms
 *
 * Arguments in/out: -
 *
 * Arguments out:    Bitstr     *bs             Coded bitstream.
 *                                              Enough memory must have been
 *                                              allocated for 'bs->b'.
 *                                              'bitioX'-Routines are used,
 *                                              i.e. the buffersize is not
 *                                              checked.
 *                   Picture    *decPict_p      resulting decoded picture.
 *                   MVField    mvOut           resulting motion vector field
 *                                              and macroblock modes.
 *                   int        *mbInd          In this field the start of
 *                                              the MBs in the bitstream are
 *                                              stored. If mbInd == NULL,
 *                                              the field is disregarded.
 *
 * Return values:    int        number of bits used.
 *
 * Example:          -
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         CodePyraLayerQ, EncodeH263
 *
 * Modified:         -
 *
 *****************************************************************************/
int CodePyraLayer(Picture *origPict, Picture *prevOrigPict,
		  Picture *spatPredPict,
		  Picture *tempPredPict, Byte *interpol[3], MVField mvIn,
		  int rate, int codingType, int scalType, int gfid, int t,
		  Bitstr *bs, Picture *decPict, MVField mvOut, int *mbInd,
		  int baseLayer_f, int codingTime)
/***********************************************************CommentEnd********/
{
  static int   q = 20;
  const int    bsIndStart = bs->ind;
  int          mx, my, mn = 0, k, gob;
  int          w = origPict->w, h = origPict->h;
  int          mbs_in_row, mb_rows_in_gob, ngobs = 18;
  int          range;
  int          mode;
#ifdef RD_MODE_DECISION
  int          lambda = 0;
#endif
  int          me_lambda = 0;
  int          mv_rate;
  int          sadI, sad0, sad;
  int          minQ = 1, maxQ = 31;
  int          scale = SCALE_MIN + q * SCALE_STEP;
  int          dmvx, dmvy;
  int          minX, minY, maxX, maxY;
  MVector      mv, pv;
  static Picture *mcPict = NULL;


  printf("CodePyraLayer ist nicht mehr up-to-date!\n");
  printf("Use CodePyraLayerQ!\n");
  return(0);

  mv.sF = 2;
  pv.sF = 2;

  /* Number of MB lines per GOB */
  mbs_in_row = w / MACROBLOCK_SIZE;
  if (h < 402) {
    mb_rows_in_gob = 1;
  } else if (h < 802) {
    mb_rows_in_gob = 2;
  } else {
    mb_rows_in_gob = 4;
  }
  ngobs = h / (MACROBLOCK_SIZE * mb_rows_in_gob);

  if (mbInd != NULL)
    mbInd[0] = bs->ind;

  if (mcPict == NULL) {
    mcPict = ClonePicture(decPict);
  } else if (mcPict->w != w || mcPict->h != h) {
    FreePicture(mcPict);
    mcPict = ClonePicture(decPict);
  }

  switch (codingType) {
  case PICTURE_CODING_TYPE_INTRA:
    do {
      printf("              scale = %2d"
	     "\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b", scale);
      fflush(stdout);
      bs->ind = bsIndStart;
      /* Send picture header */
      PyraPutPictureHeader(t, origPict->w, origPict->h,
			   codingType, scalType, q, bs);

      /**************/
      /*  GOB loop  */
      /**************/
      for (my = 0, gob = 0, mn = 0; gob < ngobs; gob++) {

	/* Send GOB header */
	if (my > 0) {
	  if (mbInd != NULL)
	    mbInd[mn] = bs->ind;
	  PyraPutGOBHeader(gob, gfid, q, bs);
	}

	for (k = 0; k < mb_rows_in_gob; my += MACROBLOCK_SIZE, k++) {
	  for (mx = 0; mx < w; mx += MACROBLOCK_SIZE, mn++) {

	    if ((k || mx > 0) && mbInd != NULL)
	      mbInd[mn] = bs->ind;

	    mvOut.mode[mn] = MODE_INTRA;
	    mvOut.mx[mn] = 0;
	    mvOut.my[mn] = 0;

	    if (baseLayer_f) {
	      /**************/
	      /* Code Intra */
	      /**************/
#ifdef PYRA_DCT_BASELAYER
	      H263dctMbEncodeIFrame(mx, my, q, origPict, decPict, bs);
#else
	      Code16x16IntraBlock(origPict, decPict, mx, my, ISCALE1(scale),
				  ISCALE0(scale), bs);
#endif
	    } else {
	      /*********************************/
	      /* Code spatial prediction error */
	      /*********************************/
	      Code16x16PredBlock(origPict, spatPredPict, decPict, mx, my,
				 scale, PICTURE_CODING_TYPE_INTRA, MODE_INTRA,
				 0, 0, lambda, bs);
	    }
	  }
	}
	/* Stuffing */
	ByteAlign0X(bs);
      }

      if ((bs->ind - bsIndStart) - rate > 0) {
	minQ = q;
	q++;
      } else {
	maxQ = q;
	q--;
      }
      scale = SCALE_MIN + q * SCALE_STEP;
    } while (q > minQ && q < maxQ);
    q = CLIP(q, 1, 31);
    scale = SCALE_MIN + q * SCALE_STEP;

    if (q != minQ) {
      bs->ind = bsIndStart;
      /* Send picture header */
      PyraPutPictureHeader(t, origPict->w, origPict->h,
			   codingType, scalType, q, bs);

      /**************/
      /*  GOB loop  */
      /**************/
      for (my = 0, gob = 0, mn = 0; gob < ngobs; gob++) {

	/* Send GOB header */
	if (my > 0) {
	  if (mbInd != NULL)
	    mbInd[mn] = bs->ind;
	  PyraPutGOBHeader(gob, gfid, q, bs);
	}

	for (k = 0; k < mb_rows_in_gob; my += MACROBLOCK_SIZE, k++) {
	  for (mx = 0; mx < w; mx += MACROBLOCK_SIZE, mn++) {

	    if ((k || (mx > 0)) && mbInd != NULL)
	      mbInd[mn] = bs->ind;

	    mvOut.mode[mn] = MODE_INTRA;
	    mvOut.mx[mn] = 0;
	    mvOut.my[mn] = 0;

	    if (baseLayer_f) {
	      /**************/
	      /* Code Intra */
	      /**************/
#ifdef PYRA_DCT_BASELAYER
	      H263dctMbEncodeIFrame(mx, my, q, origPict, decPict, bs);
#else
	      Code16x16IntraBlock(origPict, decPict, mx, my, ISCALE1(scale),
				  ISCALE0(scale),bs);
#endif
	    } else {
	      /*********************************/
	      /* Code spatial prediction error */
	      /*********************************/
	      Code16x16PredBlock(origPict, spatPredPict, decPict, mx, my,
				 scale, PICTURE_CODING_TYPE_INTRA, MODE_INTRA,
				 0, 0, lambda, bs);
	    }
	  }
	}
	/* Stuffing */
	ByteAlign0X(bs);
      }
    }
    break;
  case PICTURE_CODING_TYPE_INTER:

    if (codingTime < TIME_SUBSAMPLE_SAD)
      subsampleSAD = 2;
    else
      subsampleSAD = 1;

    /**************/
    /*  GOB loop  */
    /**************/
    for (my = 0, gob = 0, mn = 0; gob < ngobs; gob++) {

      for (k = 0; k < mb_rows_in_gob; my += MACROBLOCK_SIZE, k++) {
	for (mx = 0; mx < w; mx += MACROBLOCK_SIZE, mn++) {

	  /* Set limits for motion vectors */
	  minX = -mx;
	  maxX = tempPredPict->w - 16 - mx;
	  minY = -my;
	  maxY = tempPredPict->h - 16 - my;

	  if (baseLayer_f) {
	    /*****************************************************/
	    /* Motion estimation and mode decision for baselayer */
	    /*****************************************************/
	    sadI = MacroBlockPictureIntraSAD(mx, my, origPict, INT_MAX);
	    H263MVPrediction(&mvOut, mx >> 4, my >> 4, k == 0, &pv);
	    range = SEARCH_RANGE;
	  } else {
	    /************************************************************/
	    /* Motion estimation and mode decision for enhancementlayer */
	    /************************************************************/
	    sadI = MacroBlockPictureSAD(mx,my, origPict,spatPredPict, INT_MAX);

	    pv.x = (mvIn.mx[mn] < 0) ? (mvIn.mx[mn] - 1) / 2 : mvIn.mx[mn]/2;
	    pv.sx = ABS(mvIn.mx[mn] % 2);
	    pv.y = (mvIn.my[mn] < 0) ? (mvIn.my[mn] - 1) / 2 : mvIn.my[mn]/2;
	    pv.sy = ABS(mvIn.my[mn] % 2);

	    range = REFINEMENT_RANGE;
	  }
	  /* Clip the predictor if it points outside the image */
	  ClipMVector(minX, maxX, minY, maxY, &pv);
	  CopyMVector(&pv, &mv);
	  mv.sx = 0;
	  mv.sy = 0;

	  /* Compute SAD 0 */
	  sad0 = MacroBlockPictureSAD(mx, my, origPict, tempPredPict, INT_MAX);
	  if (me_lambda == 0)
	    sad = sad0 - SAD0_PREFERENCE;
	  else
	    sad = sad0 + me_lambda * GetMVDRate(&pv, NULL);
	  /* Motion vector refinement */
	  if (codingTime < TIME_FULLSEARCH_ME && baseLayer_f)
	    sad = FastFullPelMotionEstimationMB(mx, my, origPict, prevOrigPict,
						sad, me_lambda, &pv, &mv);
	  else
	    FullSearchFullPelME(mx, my, &sad, me_lambda, range, -15, 15,
				origPict, prevOrigPict, &pv, &mv);

	  if (me_lambda == 0 && sad0 - SAD0_PREFERENCE <= sad) {
	    sad = sad0 - SAD0_PREFERENCE;
	    ResetMVector(&mv);
	  }
	  mv_rate = me_lambda * GetMVDRate(&pv, &mv);
	  sad -= mv_rate;
	  if (sadI < sad - INTER_PREFERENCE)
	    mode = MODE_INTRA;
	  else
	    mode = MODE_INTER;


	  mvOut.mode[mn] = mode;
	  switch (mode) {
	  case MODE_INTRA:
	    /* Reset motion vector */
	    mvOut.mx[mn] = 0;
	    mvOut.my[mn] = 0;
	    break;
	  case MODE_INTER:
	    /* Half pel refinement */
	    sad += mv_rate;
	    sad = HalfPelMotionEstimationMB(mx, my, origPict, interpol,
					    mv.x > -16, 1, mv.y > -16, 1,
					    sad, me_lambda, &pv, &mv);
	    /* Motion compensate macroblock */
	    MCMacroBlockPicture(mx, my, tempPredPict, interpol, &mv, mcPict);
	    /* Attention!! 'decCImg' now holds the motion compensated
	       temporal prediction image.
	       This is a little bit dirty but prevents the allocation of
	       another picture */

	    /* Store motion vector */
	    mvOut.mx[mn] = 2 * mv.x + mv.sx;
	    mvOut.my[mn] = 2 * mv.y + mv.sy;

	    break;
	  }
	}
      }
    }

    do {
      printf("              scale = %2d"
	     "\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b\b", scale);
      fflush(stdout);
      bs->ind = bsIndStart;

      /* Send picture header */
      PyraPutPictureHeader(t, origPict->w, origPict->h, codingType,
			   scalType, q, bs);

      /**************/
      /*  GOB loop  */
      /**************/
      for (my = 0, gob = 0, mn = 0; gob < ngobs; gob++) {

	/* Send GOB header */
	if (my > 0) {
	  if (mbInd != NULL)
	    mbInd[mn] = bs->ind;
	  PyraPutGOBHeader(gob, gfid, q, bs);
	}

	for (k = 0; k < mb_rows_in_gob; my += MACROBLOCK_SIZE, k++) {
	  for (mx = 0; mx < w; mx += MACROBLOCK_SIZE, mn++) {

	    if ((k || (mx > 0)) && mbInd != NULL)
	      mbInd[mn] = bs->ind;

	    switch (mvOut.mode[mn]) {
	    case MODE_INTRA:
	      if (baseLayer_f) {
		/**************/
		/* Code Intra */
		/**************/
		PutMode(2, bs); /* coded intra block */
#ifdef PYRA_DCT_BASELAYER
		H263dctMbEncodeIFrame(mx, my, q, origPict, decPict, bs);
#else
		Code16x16IntraBlock(origPict, decPict, mx, my, ISCALE1(scale),
				    ISCALE0(scale), bs);
#endif
	      } else {
		/*********************************/
		/* Code spatial prediction error */
		/*********************************/
		Code16x16PredBlock(origPict, spatPredPict, decPict, mx, my,
				   scale, PICTURE_CODING_TYPE_INTER,
				   MODE_INTRA, 0, 0, lambda, bs);
	      }
	      break;
	    case MODE_INTER:
	      /**************/
	      /* Code Inter */
	      /**************/
	      /* Set limits for motion vectors */
	      minX = -mx;
	      maxX = tempPredPict->w - 16 - mx;
	      minY = -my;
	      maxY = tempPredPict->h - 16 - my;
	      /* Compute motion vector difference for sending */
	      if (baseLayer_f) {
		H263MVPrediction(&mvOut, mx >> 4, my >> 4, k == 0, &pv);
	      } else {
		pv.x = (mvIn.mx[mn] < 0) ? (mvIn.mx[mn]-1) / 2 : mvIn.mx[mn]/2;
		pv.sx = ABS(mvIn.mx[mn] % 2);
		pv.y = (mvIn.my[mn] < 0) ? (mvIn.my[mn]-1) / 2 : mvIn.my[mn]/2;
		pv.sy = ABS(mvIn.my[mn] % 2);
	      }
	      /* Clip the predictor if it points outside the image */
	      ClipMVector(minX, maxX, minY, maxY, &pv);
	      dmvx = mvOut.mx[mn] - (2 * pv.x + pv.sx);
	      dmvy = mvOut.my[mn] - (2 * pv.y + pv.sy);
	      /* Clip motion vector difference */
	      if (dmvx > 31)
		dmvx -= 64;
	      else if (dmvx < -32)
		dmvx += 64;
	      if (dmvy > 31)
		dmvy -= 64;
	      else if (dmvy < -32)
		dmvy += 64;

	      Code16x16PredBlock(origPict, mcPict, decPict, mx, my, scale,
				 PICTURE_CODING_TYPE_INTER, MODE_INTER,
				 dmvx, dmvy, lambda, bs);
	      /* Store motion vector as predictor for the next MB */
	      break;
	    }
	  }
	}
	/* Stuffing */
	ByteAlign0X(bs);
      }
      if ((bs->ind - bsIndStart) - rate > 0) {
	minQ = q;
	q++;
      } else {
	maxQ = q;
	q--;
      }
      scale = SCALE_MIN + q * SCALE_STEP;
    } while (q > minQ && q < maxQ);
    q = CLIP(q, 1, 31);
    scale = SCALE_MIN + q * SCALE_STEP;

    if (q != minQ) {

      bs->ind = bsIndStart;

      /* Send picture header */
      PyraPutPictureHeader(t, origPict->w, origPict->h, codingType,
			   scalType, q, bs);

      /**************/
      /*  GOB loop  */
      /**************/
      for (my = 0, gob = 0, mn = 0; gob < ngobs; gob++) {

	/* Send GOB header */
	if (my > 0) {
	  if (mbInd != NULL)
	    mbInd[mn] = bs->ind;
	  PyraPutGOBHeader(gob, gfid, q, bs);
	}
#ifdef PROTOCOL
	if (fp_prot_g != NULL && protFormat_g >= 3)
	  fprintf(fp_prot_g,"\n");
#endif

	for (k = 0; k < mb_rows_in_gob; my += MACROBLOCK_SIZE, k++) {
	  for (mx = 0; mx < w; mx += MACROBLOCK_SIZE, mn++) {

	    if ((k || (mx > 0)) && mbInd != NULL)
	      mbInd[mn] = bs->ind;

	    switch (mvOut.mode[mn]) {
	    case MODE_INTRA:
	      if (baseLayer_f) {
		/**************/
		/* Code Intra */
		/**************/
#ifdef PROTOCOL
		if (fp_prot_g != NULL && protFormat_g >= 3)
		  fprintf(fp_prot_g,"I");
#endif
		PutMode(2, bs); /* coded intra block */
#ifdef PYRA_DCT_BASELAYER
		H263dctMbEncodeIFrame(mx, my, q, origPict, decPict, bs);
#else
		Code16x16IntraBlock(origPict, decPict, mx, my, ISCALE1(scale),
				    ISCALE0(scale), bs);
#endif
	      } else {
		/*********************************/
		/* Code spatial prediction error */
		/*********************************/
#ifdef PROTOCOL
		if (fp_prot_g != NULL && protFormat_g >= 3)
		  fprintf(fp_prot_g,"S");
#endif
		Code16x16PredBlock(origPict, spatPredPict, decPict, mx, my,
				   scale, PICTURE_CODING_TYPE_INTER,
				   MODE_INTRA, 0, 0, lambda, bs);
	      }
	      break;
	    case MODE_INTER:
	      /**************/
	      /* Code Inter */
	      /**************/

	      /* Set limits for motion vectors */
	      minX = -mx;
	      maxX = tempPredPict->w - 16 - mx;
	      minY = -my;
	      maxY = tempPredPict->h - 16 - my;
	      /* Compute motion vector difference for sending */
	      if (baseLayer_f) {
		H263MVPrediction(&mvOut, mx >> 4, my >> 4, k == 0, &pv);
	      } else {
		pv.x = (mvIn.mx[mn] < 0) ? (mvIn.mx[mn]-1) / 2 : mvIn.mx[mn]/2;
		pv.sx = ABS(mvIn.mx[mn] % 2);
		pv.y = (mvIn.my[mn] < 0) ? (mvIn.my[mn]-1) / 2 : mvIn.my[mn]/2;
		pv.sy = ABS(mvIn.my[mn] % 2);
	      }
	      /* Clip the predictor if it points outside the image */
	      ClipMVector(minX, maxX, minY, maxY, &pv);
	      dmvx = mvOut.mx[mn] - (2 * pv.x + pv.sx);
	      dmvy = mvOut.my[mn] - (2 * pv.y + pv.sy);
	      /* Clip motion vector difference */
	      if (dmvx > 31)
		dmvx -= 64;
	      else if (dmvx < -32)
		dmvx += 64;
	      if (dmvy > 31)
		dmvy -= 64;
	      else if (dmvy < -32)
		dmvy += 64;
#ifdef PROTOCOL
	      if (fp_prot_g != NULL && protFormat_g >= 3)
		fprintf(fp_prot_g,"T");
#endif
	      Code16x16PredBlock(origPict, mcPict, decPict, mx, my, scale,
				 PICTURE_CODING_TYPE_INTER, MODE_INTER,
				 dmvx, dmvy, lambda, bs);
	      break;
	    }
	  }
	}
	/* Stuffing */
	ByteAlign0X(bs);
      }
    }
    break;
  default:
    fprintf(stderr, "CodePyraLayer: Coding type >%d< not supported!\n",
	    codingType);
    return(-1);
  }


  /* Return number of coded bits */
  return(bs->ind - bsIndStart);
}



/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- CodePyraLayerQ -- Code one layer of the pyramid with VQ using fixed scale
 *
 * Author:           K.S.
 *
 * Created:          1-Sep-97
 *
 * Purpose:          One layer of the pyramid is coded as described in Uwe's
 *                   Dis. The coding is done with the given fixed scale.
 *
 * Arguments in:     unsigned char *crvec       field that containes
 *                                              at 'crvec[mn] & 0x80' which MBs
 *                                              are changed.
 *                                              Disregarded if NULL.
 *                   Picture    *origPict       original image
 *                                              (possibly downsampled)
 *                   Picture    *prevOrigPict   previous original image
 *                   Picture    *spatPredPict   spatial prediction image
 *                                              (decoded image from lower
 *                                              layer).
 *                   Picture    *tempPredPict   temporal prediction image
 *                                              (last decoded image from this
 *                                               layer).
 *                   Byte       *interpol[3]    Halfpel interpolated versions
 *                                              of 'tempPredPict'.
 *                   MVField    mvIn            Input motion vector field
 *                                              (MVField from lower layer).
 *                   int        q               scale for VQ is
 *                                              SCALE_MIN + q * SCALE_STEP.
 *                   int        codingType      PICTURE_CODING_TYPE_INTRA or
 *                                              PICTURE_CODING_TYPE_INTER.
 *                   int        scalType        scalability type.
 *                                               0: spatial (and temp.) scal.
 *                                               1: temporal scalability
 *                                               1: SNR scalability
 *                   int        t               time reference to be put into
 *                                              the bitstream.
 *                   int        baseLayer_f     1 is this is the base layer,
 *                                              0 otherwise.
 *                   int        codingTime      0... select between fast and
 *                                              and slow/good coding algorithms
 *
 * Arguments in/out: -
 *
 * Arguments out:    Bitstr     *bs             Coded bitstream.
 *                                              Enough memory must have been
 *                                              allocated for 'bs->b'.
 *                                              'bitioX'-Routines are used,
 *                                              i.e. the buffersize is not
 *                                              checked.
 *                   Picture    *decPict        resulting decoded picture.
 *                   MVField    mvOut           resulting motion vector field
 *                                              and macroblock modes.
 *                   int        *mbInd          In this field the start of
 *                                              the MBs in the bitstream are
 *                                              stored. If mbInd == NULL,
 *                                              the field is disregarded.
 *
 * Return values:    int        number of bits used.
 *
 * Example:          -
 *
 * Side effects:     -
 *
 * Description:      -
 *
 * See also:         CodePyraLayer, EncodeH263Q
 *
 * Modified:         -
 *
 *****************************************************************************/
int CodePyraLayerQ(unsigned char *crvec, Picture *origPict, Picture *prevOrigPict,
		   Picture *spatPredPict,
		   Picture *tempPredPict, Byte *interpol[3], MVField mvIn,
		   int q, int codingType, int scalType, int gfid, int t,
		   Bitstr *bs, Picture *decPict, MVField mvOut, int *mbInd,
		   int baseLayer_f, int codingTime)
/***********************************************************CommentEnd********/
{
  const int  bsIndStart = bs->ind;
  int        bsInd;
  int        scale = SCALE_MIN + q * SCALE_STEP;
  int        mx, my, mn = 0, k, gob;
  int        w = origPict->w, h = origPict->h;
  int        mbs_in_row, mb_rows_in_gob, ngobs = 18;
  int        range;
  int        mode;
  int        sadI, sad0, sad;
#ifdef RD_MODE_DECISION
  int        costU, costI, costP, costMin, modeMin;
#endif
  int        lambda = 0;
  int        me_lambda = q;
  int        mv_rate;
  int        dmvx, dmvy;
  int        minX, minY, maxX, maxY;
  MVector    mv, pv;
#if 0
  Byte       mblk[6*64];   /* for storing the decoded I macroblock */
#endif


  ASSERT(q > 0 && q <= 31);

#ifdef RD_MODE_DECISION
  if (codingTime >= TIME_RD_MODE_DECISION)
    lambda = (17 * SQR(q)) / 20;
#endif

  mv.sF = 2;
  pv.sF = 2;

  /* Number of MB lines per GOB */
  mbs_in_row = w / MACROBLOCK_SIZE;
  if (h < 402) {
    mb_rows_in_gob = 1;
  } else if (h < 802) {
    mb_rows_in_gob = 2;
  } else {
    mb_rows_in_gob = 4;
  }
  ngobs = h / (MACROBLOCK_SIZE * mb_rows_in_gob);

  if (mbInd != NULL)
    mbInd[0] = bs->ind;

  /* Send picture header */
  PyraPutPictureHeader(t, origPict->ws, origPict->hs, codingType,
		       scalType, q, bs);

  switch (codingType) {
  case PICTURE_CODING_TYPE_INTRA:

#ifndef PYRA_DCT_BASELAYER
    if (baseLayer_f) {
      EncodePictDPCM(origPict, decPict, bs);
      /* Stuffing */
#ifdef COUNT_BITS
      {
	int n =
#endif
	  ByteAlign0X(bs);
#ifdef COUNT_BITS
	bitCountP->totalBits += n;
	bitCountP->headerBits += n;
      }
#endif
      break;
    }
#endif

    /**************/
    /*  GOB loop  */
    /**************/
    for (my = 0, gob = 0, mn = 0; gob < ngobs; gob++) {

      /* Send GOB header */
      if (my > 0) {
	if (mbInd != NULL)
	  mbInd[mn] = bs->ind;
	PyraPutGOBHeader(gob, gfid, q, bs);
      }

#ifdef PROTOCOL
      if (fp_prot_g != NULL && protFormat_g >= 3)
	fprintf(fp_prot_g,"\n");
#endif

      for (k = 0; k < mb_rows_in_gob; my += MACROBLOCK_SIZE, k++) {
	for (mx = 0; mx < w; mx += MACROBLOCK_SIZE, mn++) {

	  if ((k || (mx > 0)) && mbInd != NULL)
	    mbInd[mn] = bs->ind;

	  mvOut.mode[mn] = MODE_INTRA;
	  mvOut.mx[mn] = 0;
	  mvOut.my[mn] = 0;

	  if (baseLayer_f) {
	    /**************/
	    /* Code Intra */
	    /**************/
#ifdef PYRA_DCT_BASELAYER
	    H263dctMbEncodeIFrame(mx, my, q, origPict, decPict, bs);
#else
	    Code16x16IntraBlock(origPict, decPict, mx, my, ISCALE1(scale),
				ISCALE0(scale), bs);
#endif
	  } else {
	    /*********************************/
	    /* Code spatial prediction error */
	    /*********************************/
	    Code16x16PredBlock(origPict, spatPredPict, decPict, mx, my,
			       scale, PICTURE_CODING_TYPE_INTRA, MODE_INTRA,
			       0, 0, lambda, bs);
	  }
#ifdef PROTOCOL
	  if (fp_prot_g != NULL && protFormat_g >= 3) {
	    if (baseLayer_f)
	      fprintf(fp_prot_g,"I");
	    else
	      fprintf(fp_prot_g,"S");
	  }
#endif
	}
      }
      /* Stuffing */
#ifdef COUNT_BITS
      {
	int n =
#endif
      ByteAlign0X(bs);
#ifdef COUNT_BITS
	bitCountP->totalBits += n;
	bitCountP->headerBits += n;
      }
#endif
    }
    break;

  case PICTURE_CODING_TYPE_INTER:

    if (codingTime < TIME_SUBSAMPLE_SAD_4)
      subsampleSAD = 4;
    else if (codingTime < TIME_SUBSAMPLE_SAD)
      subsampleSAD = 2;
    else
      subsampleSAD = 1;

    /**************/
    /*  GOB loop  */
    /**************/
    for (my = 0, gob = 0, mn = 0; gob < ngobs; gob++) {

      /* Send GOB header */
      if (my > 0) {
	if (mbInd != NULL)
	  mbInd[mn] = bs->ind;
	PyraPutGOBHeader(gob, gfid, q, bs);
      }

#ifdef PROTOCOL
      if (fp_prot_g != NULL && protFormat_g >= 3)
	fprintf(fp_prot_g,"\n");
#endif

      for (k = 0; k < mb_rows_in_gob; my += MACROBLOCK_SIZE, k++) {
	for (mx = 0; mx < w; mx += MACROBLOCK_SIZE, mn++) {

	  if ((k || (mx > 0)) && mbInd != NULL)
	    mbInd[mn] = bs->ind;

	  /* Conditional replenishment */
	  if (crvec != NULL)
	    if (!(crvec[mn] & 0x80)) {  /* UNCODED */
	      /* Store mode */
	      mvOut.mode[mn] = MODE_UNCODED;
	      mvOut.mx[mn] = 0;
	      mvOut.my[mn] = 0;
	      PutMode(0, bs);   /* Inter, uncoded temp. */
	      CopyMacroBlockPicture(mx, my, tempPredPict, decPict);
#ifdef PROTOCOL
	      if (fp_prot_g != NULL && protFormat_g >= 3)
		fprintf(fp_prot_g,"U");
#endif
	      continue;
	    }

#ifdef RD_MODE_DECISION
	  if (lambda) {
	    costU = INT_MAX;
	  }
#endif

	  /* Set limits for motion vectors */
	  minX = -mx;
	  maxX = tempPredPict->w - 16 - mx;
	  minY = -my;
	  maxY = tempPredPict->h - 16 - my;

	  if (baseLayer_f) {
	    /*****************************************************/
	    /* Motion estimation and mode decision for baselayer */
	    /*****************************************************/
	    sadI = MacroBlockPictureIntraSAD(mx, my, origPict, INT_MAX);
	    H263MVPrediction(&mvOut, mx >> 4, my >> 4, k == 0, &pv);
	    range = SEARCH_RANGE;
	  } else {
	    /************************************************************/
	    /* Motion estimation and mode decision for enhancementlayer */
	    /************************************************************/
	    sadI = MacroBlockPictureSAD(mx,my, origPict,spatPredPict, INT_MAX);

	    pv.x = (mvIn.mx[mn] < 0) ? (mvIn.mx[mn] - 1) / 2 : mvIn.mx[mn]/2;
	    pv.sx = ABS(mvIn.mx[mn] % 2);
	    pv.y = (mvIn.my[mn] < 0) ? (mvIn.my[mn] - 1) / 2 : mvIn.my[mn]/2;
	    pv.sy = ABS(mvIn.my[mn] % 2);

	    range = REFINEMENT_RANGE;
	  }
	  /* Clip the predictor if it points outside the image */
	  ClipMVector(minX, maxX, minY, maxY, &pv);
	  CopyMVector(&pv, &mv);
	  mv.sx = 0;
	  mv.sy = 0;

	  /* Motion vector refinement or motion estimation */
	  if (me_lambda == 0)
	    sad = sadI;
	  else
	    sad = INT_MAX;
	  if (codingTime < TIME_FULLSEARCH_ME && baseLayer_f)
	    sad = FastFullPelMotionEstimationMB(mx, my, origPict, prevOrigPict,
						sad, me_lambda, &pv, &mv);
	  else
	    FullSearchFullPelME(mx, my, &sad, me_lambda, range, -15, 15,
				origPict, prevOrigPict, &pv, &mv);

	  if (me_lambda != 0) {
	    mv_rate = me_lambda * GetMVDRate(&pv, &mv);
	    sad -= mv_rate;
	  } else
	    mv_rate = 0;

	  sad = FPMotionMacroBlockPictureSAD(mx, my, origPict, tempPredPict,
					     &mv, INT_MAX);

	  if (sadI < (sad - /*(baseLayer_f ?*/ 500/* : 0)*/) || lambda)
	    mode = MODE_INTRA;
	  else
	    mode = MODE_INTER;

	  switch (mode) {
	  case MODE_INTRA:
	    if (baseLayer_f) {
	      /**************/
	      /* Code Intra */
	      /**************/
#ifndef RD_MODE_DECISION
#ifdef PROTOCOL
	      if (fp_prot_g != NULL && protFormat_g >= 3)
		fprintf(fp_prot_g,"I");
#endif
#endif
	      bsInd = bs->ind;
	      PutMode(2, bs); /* coded intra block */
#ifdef PYRA_DCT_BASELAYER
	      H263dctMbEncodeIFrame(mx, my, q, origPict, decPict, bs);
#else
	      Code16x16IntraBlock(origPict, decPict, mx, my, ISCALE1(scale),
				  ISCALE0(scale), bs);
#endif
	    } else {
	      /*********************************/
	      /* Code spatial prediction error */
	      /*********************************/
#ifdef PROTOCOL
	      if (fp_prot_g != NULL && protFormat_g >= 3)
		fprintf(fp_prot_g,"S");
#endif
	      bsInd = bs->ind;
	      Code16x16PredBlock(origPict, spatPredPict, decPict, mx, my,
				 scale, PICTURE_CODING_TYPE_INTER, MODE_INTRA,
				 0, 0, lambda, bs);
	    }
	    mvOut.mx[mn] = 0;
	    mvOut.my[mn] = 0;

#ifdef RD_MODE_DECISION
	    if (lambda) {
	      costI = lambda * (bs->ind - bsInd);
	      costI += MacroBlockPictureSSE(mx, my, origPict, decPict,
					    costU - costI);
	      bs->ind = bsInd;     /* Reset bitstream */
	      if (costI < costU) {
		costMin = costI;
		modeMin = MODE_INTRA;
		/*ExtractMacroBlock(mx, my, decPict, mblk);*/
	      } else {
		costMin = costU;
		modeMin = MODE_UNCODED;
	      }
	    } else
#endif
	      break;
	  case MODE_INTER:
	    /**************/
	    /* Code Inter */
	    /**************/

	    /*sad = FPMotionMacroBlockPictureSAD(mx, my, origPict, tempPredPict,
					       &mv, INT_MAX);*/
	    if (me_lambda != 0)
	      sad += mv_rate;
	    else {
	      /* Compute SAD 0 */
	      if (baseLayer_f)
		sad0 = MacroBlockPictureSAD(mx,my,origPict,tempPredPict,
					    INT_MAX);
	      else
		sad0 = FPMotionMacroBlockPictureSAD(mx,my,origPict,
						    tempPredPict,
						    &pv,INT_MAX) - 100;
	    }

	    /* Half pel refinement */
	    /*sad += mv_rate;*/
	    sad = HalfPelMotionEstimationMB(mx, my, origPict, interpol,
					    mv.x > -16, 1, mv.y > -16, 1,
					    sad, me_lambda, &pv, &mv);
	    if (me_lambda == 0) {
	      if (sad0 < sad)
		if (baseLayer_f)
		  ResetMVector(&mv);
		else
		  CopyMVector(&pv, &mv);
	    }

	    /* Motion compensate macroblock */
	    MCMacroBlockPicture(mx, my, tempPredPict, interpol, &mv, decPict);
	    /* Attention!! 'decCImg' now holds the motion compensated
	       temporal prediction image.
	       This is a little bit dirty but prevents the allocation of
	       another picture */

	    /* Store motion vector */
	    mvOut.mx[mn] = 2 * mv.x + mv.sx;
	    mvOut.my[mn] = 2 * mv.y + mv.sy;
	    /* Compute motion vector difference for sending */
	    dmvx = 2 * (mv.x - pv.x) + mv.sx - pv.sx;
	    dmvy = 2 * (mv.y - pv.y) + mv.sy - pv.sy;
	    /* Clip motion vector difference */
	    if (dmvx > 31)
	      dmvx -= 64;
	    else if (dmvx < -32)
	      dmvx += 64;
	    if (dmvy > 31)
	      dmvy -= 64;
	    else if (dmvy < -32)
	      dmvy += 64;

#ifndef RD_MODE_DECISION
#ifdef PROTOCOL
	    if (fp_prot_g != NULL && protFormat_g >= 3)
	      fprintf(fp_prot_g,"T");
#endif
#endif
	    bsInd = bs->ind;
	    Code16x16PredBlock(origPict, decPict, decPict, mx, my, scale,
			       PICTURE_CODING_TYPE_INTER, MODE_INTER,
			       dmvx, dmvy, lambda, bs);

#ifdef RD_MODE_DECISION
	    /* Rate Distortion Mode Decision */
	    if (lambda) {
	      costP = lambda * (bs->ind - bsInd);
	      costP += MacroBlockPictureSSE(mx, my, origPict, decPict,
					    costMin - costP);
	      if (costP >= costMin) {   /* P is not the best mode */
		bs->ind = bsInd;    /* Reset bitstream */
		mode = modeMin;
		mvOut.mx[mn] = 0;
		mvOut.my[mn] = 0;
		if (modeMin == MODE_UNCODED) {
		  PutMode(0, bs);   /* Inter, uncoded temp. */
		  CopyMacroBlockPicture(mx, my, tempPredPict, decPict);
		  /*bits = H263putPFrameMB(mb, dq, bs);*/
#ifdef PROTOCOL
		  if (fp_prot_g != NULL && protFormat_g >= 3)
		    fprintf(fp_prot_g,"U");
#endif
		} else if (modeMin == MODE_INTRA || modeMin == MODE_INTRA_Q) {
		  if (baseLayer_f) {
		    PutMode(2, bs); /* coded intra block */
#ifdef PYRA_DCT_BASELAYER
		    H263dctMbEncodeIFrame(mx, my, q, origPict, decPict, bs);
#else
		    Code16x16IntraBlock(origPict,decPict,mx,my,ISCALE1(scale),
					ISCALE0(scale), bs);
#endif
#ifdef PROTOCOL
		    if (fp_prot_g != NULL && protFormat_g >= 3)
		      fprintf(fp_prot_g,"I");
#endif
		  } else {
		    Code16x16PredBlock(origPict, spatPredPict, decPict, mx, my,
				       scale, PICTURE_CODING_TYPE_INTER,
				       MODE_INTRA, 0, 0, lambda, bs);
#ifdef PROTOCOL
		    if (fp_prot_g != NULL && protFormat_g >= 3)
		      fprintf(fp_prot_g,"S");
#endif
		  }
		}
	      } else {
		mode = MODE_INTER;
	      }
	    }
#endif
	    break;
	  }

	  /* Store mode */
	  mvOut.mode[mn] = mode;

#ifdef PROTOCOL
	  if (fp_prot_g != NULL && protFormat_g >= 3) {
	    if (mode == MODE_INTER)
	      fprintf(fp_prot_g,"T");
	  }
#endif

	}
      }
      /* Stuffing */
#ifdef COUNT_BITS
      {
	int n =
#endif
      ByteAlign0X(bs);
#ifdef COUNT_BITS
	bitCountP->totalBits += n;
	bitCountP->headerBits += n;
      }
#endif
    }
#ifdef PROTOCOL
    if (fp_prot_g != NULL && protFormat_g >= 4) {
      fprintf(fp_prot_g,"\n");
      PrintMVField(&mvOut, fp_prot_g);
      fprintf(fp_prot_g,"\n\n");
    }
#endif
    break;
  default:
    fprintf(stderr, "CodePyraLayerQ: Coding type >%d< not supported!\n",
	    codingType);
    return(-1);
  }

  /* Return number of coded bytes */
  return(bs->ind - bsIndStart);
}


int CodePyraLayerQIntra8x8(Picture *origPict, Picture *spatPredPict, int q, int gfid, int t,
		   Bitstr *bs, Picture *decPict)
{
  const int  bsIndStart = bs->ind;
  int        scale = SCALE_MIN + q * SCALE_STEP;
  int mx, my;

  scale = 18; /* Test, test, test ... */
  for (my = 0; my<72; my += 8) {
    for (mx = 0; mx<88; mx += 8) {

      Code8x8PredBlock(origPict, spatPredPict, decPict, mx, my,
		       scale, PICTURE_CODING_TYPE_INTER, MODE_INTRA,
		       0, 0, bs);
    }

    fprintf(stderr,"\n");
  }
  return(bs->ind - bsIndStart);
}      


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PyraPutPictureHeader -- Put picture header for pyramid layer to bitstream
 *
 * Author:           K.S.
 *
 * Created:          1-Sep-97
 *
 * Purpose:          Write the picture header for one pyramid VQ layer to
 *                   the bitstream.
 *
 * Arguments in:     int        t             time reference to be put into the
 *                                            bitstream.
 *                   int        ws            (sub) image width.
 *                   int        hs            (sub) image height.
 *                   int        codingType    PICTURE_CODING_TYPE_INTRA or
 *                                            PICTURE_CODING_TYPE_INTER.
 *                   int        q             scale for VQ is
 *                                            SCALE_MIN + q * SCALE_STEP.
 *
 * Arguments in/out: -
 *
 * Arguments out:    Bitstr     *bs           Bitstream.
 *                                            Enough memory must have been
 *                                            allocated for 'bs->b'.
 *                                            'bitioX'-Routines are used, i.e.
 *                                            the buffersize is not checked.
 *
 * Return values:    int        number of bits used.
 *
 * Example:          -
 *
 * Side effects:     -
 *
 * Description:      * The picture header is byte aligned by stuffing 0 - 7
 *                     zeros.
 *                   * Picture start code: (29 bits)
 *                     00000000 00000000 00000001 00000
 *                   * Time reference:  (8 bits)
 *                   * Image size: (3 bits)
 *                       010   --> SQCIF
 *                       010   --> QCIF
 *                       011   --> CIF
 *                       100   --> 4CIF
 *                       101   --> 16CIF
 *                       110   --> custom format:
 *                                  w : (11 bits)
 *                                  h : (11 bits)
 *                     Others are not implemented yet.
 *                   * Coding type: (1 bit)
 *                       0   --> PICTURE_CODING_TYPE_INTRA
 *                       1   --> PICTURE_CODING_TYPE_INTER
 *                   * Scalability type: (2 bits)  (im Moment voellig unnoetig)
 *                       00  --> spatial (and temporal) scalabilty
 *                               (i.e. motion vectors are present,
 *                                the reference picture from which the motion
 *                                prediction is taken is the last picture of
 *                                this layer).
 *                       01  --> temporal scalability
 *                               (i.e. motion vectors are present,
 *                                the reference picture from which the motion
 *                                prediction is taken is the last picture of
 *                                the next lower layer; this layer and the next
 *                                lower layer should be displayed).
 *                       10  --> SNR scalability
 *                               (i.e. no motion vectors are present).
 *                   * Scale:   (5 bits)
 *                       actually  'q = (scale - SCALE_MIN) / SCALE_STEP'
 *                       is sent.
 *
 * See also:         PyraPutGOBHeader, H263PutPictureHeader
 *
 * Modified:         -
 *
 *****************************************************************************/
int PyraPutPictureHeader(int t, int ws, int hs, int codingType, int scalType,
			 int q, Bitstr *bs)
/***********************************************************CommentEnd********/
{
  int n;


  /* Stuffing */
  n = ByteAlign0X(bs);

  /* PSC */
  PutNumberX(1, 24, bs);
  n += 24;
  PutNumberX(0, 5, bs);
  n += 5;

  /* Time reference */
  PutNumberX(t, 8, bs);
  n += 8;

  /* Image size */
  if (ws == SQCIF_WIDTH && hs == SQCIF_HEIGHT) 
    PutNumberX(1, 3, bs);
  else if (ws == QCIF_WIDTH && hs == QCIF_HEIGHT) 
    PutNumberX(2, 3, bs);
  else if (ws == CIF_WIDTH && hs == CIF_HEIGHT) 
    PutNumberX(3, 3, bs);
  else if (ws == 2 * CIF_WIDTH && hs == 2 * CIF_HEIGHT) 
    PutNumberX(4, 3, bs);
  else if (ws == 4 * CIF_WIDTH && hs == 4 * CIF_HEIGHT) 
    PutNumberX(5, 3, bs);
  else {
    PutNumberX(6, 3, bs);
    PutNumberX(ws, 11, bs);
    PutNumberX(hs, 11, bs);
    n += 22;
  }
  n += 3;

  /* Send picture coding type */
  switch (codingType) {
  case PICTURE_CODING_TYPE_INTRA:
    PutBitX(0, bs);
    break;
  case PICTURE_CODING_TYPE_INTER:
    PutBitX(1, bs);
    break;
  default:
    fprintf(stderr, "PyraPutPictureHeader: Can not send coding type %d.\n",
	    codingType);
    exit(-1);
  }
  n += 1;

#if 1
  /* Send picture coding type */
  switch (scalType) {
  case 0:   /* spatial scal. */
    PutNumberX(0, 2, bs);
    break;
  case 1:   /* Temp. scal. */
    PutNumberX(1, 2, bs);
    break;
  case 2:   /* SNR scal. */
    PutNumberX(2, 2, bs);
    break;
  default:
    fprintf(stderr, "PyraPutPictureHeader: scalability type %d not supported."
	    "\n", scalType);
    exit(-1);
  }
  n += 2;
#endif

  /* Send q (scale) */
  PutNumberX(q, 5, bs);
  n += 5;

#ifdef COUNT_BITS
    bitCountP->totalBits += n;
    bitCountP->headerBits += n;
#endif

  return(n);
}


/***********************************************************CommentBegin******
 *****************************************************************************
 *
 * -- PyraPutGOBHeader -- Put GOB header for pyramid layer to bitstream
 *
 * Author:           K.S.
 *
 * Created:          1-Sep-97
 *
 * Purpose:          Write the GOB (group of blocks) header for one
 *                   pyramid VQ layer to the bitstream.
 *
 * Arguments in:     int     nGOB    GOB number.
 *                   int     gfid    group frame ID.
 *                   int     q       scale for VQ is SCALE_MIN + q*SCALE_STEP.
 *
 * Arguments in/out: -
 *
 * Arguments out:    Bitstr  *bs     Bitstream.
 *                                   Enough memory must have been
 *                                   allocated for 'bs->b'.
 *                                   'bitioX'-Routines are used, i.e.
 *                                   the buffersize is not checked.
 *
 * Return values:    int     number of bits used.
 *
 * Example:          -
 *
 * Side effects:     -
 *
 * Description:      Attention: For GOB 0 no GOB header is sent.
 *                         Otherwise there would be a start code emulation.
 *                   The GOB header is byte aligned by stuffing 0 - 7 zeros.
 *                   * GOB Sync: (24 bits)
 *                     00000000 00000000 00000001
 *                   * GOB number: (5 bits)
 *                   * GOB frame ID:  (2 bits)
 *                   * Scale:   (5 bits)
 *                       actually  'q = (scale - SCALE_MIN) / SCALE_STEP'
 *                       is sent.
 *
 * See also:         PyraPutPictureHeader, H263PutGOBHeader
 *
 * Modified:         -
 *
 *****************************************************************************/
int PyraPutGOBHeader(int nGOB, int gfid, int q, Bitstr *bs)
/***********************************************************CommentEnd********/
{
  int n;


  /* Stuffing */
  n = ByteAlign0X(bs);

  /* GSC */
  PutNumberX(1, 24, bs);
  n += 24;
  PutNumberX(nGOB, 5, bs);
  n += 5;

  /* GOB frame ID */
  PutNumberX(gfid, 2, bs);
  n += 2;

  /* Send q (scale) */
  PutNumberX(q, 5, bs);
  n += 5;

#ifdef COUNT_BITS
    bitCountP->totalBits += n;
    bitCountP->headerBits += n;
#endif

  return(n);
}
