/* #include <string.h> */

#include <string.h>
#include <assert.h>
#include <bitOut.h>
#include "rtp_defs.h"
#include "rtpsupport.h"
#include "conceal.h"
#include "statistics.h"
#include "decisions.h"


/*--------------------------------------------------------------------------------
 * Criteria if layer should be displayed or not
 *--------------------------------------------------------------------------------
 */
void showDisplayCriteria()
{
    printf("\nDisplay criteria:\n");
    printf("\t  0: - show layer anyway ignoring losses\n");
    printf("\t 11: - if loss exceeds maximum switch off permanently\n");
    printf("\t       current layer and all enhancement layers\n");
    printf("\t 12: - if accumulated gob losses since last intra attenuated\n");
    printf("\t       by a fading factor exceed maximum switch off permanently\n");
    printf("\t       current layer and all enhancement layers\n");
    printf("\t 13: - if sum of previous 'nrl' losses exceeds maximum switch off\n");
    printf("\t       permanently current layer and all enhancement layers\n");
    printf("\t 14: - if accumulated gob losses since last intra attenuated\n");
    printf("\t       by a fading factor exceed maximum switch off permanently\n");
    printf("\t       current layer and all enhancement layers\n");
    printf("\t 21: - if current loss exceeds maximum don't display\n");
    printf("\t 22: - if sum of previous 'nrl' losses exceeds maximum\n");
    printf("\t       don't display\n");
    printf("\t 31: - if loss exceeds maximum switch off current layer and all\n");
    printf("\t       enhancement layers until next intra\n");
    putchar('\n');
}
/*
 * Compute statistics of packet loss for current layer.
 * Decide if layer should be displayed or not.
 * Returns nonzero if layer should be displayed.
 */
int considerPyraPacketLoss(layerDecData **ldd, concealDispVal *cd,
                           int nl, const unsigned long curtime)
{
    int ii, xx, addedPercentLoss= 0;
    /* int display=SHOW_LAYER; */

    xx = ldd[nl]->curIndex;
    /* check criteria */
    switch ( cd->displayCriterium )
    {
      case 0:
          /* show layer anyway ignoring losses */
          return SHOW_LAYER;
          break;          
      case 11:
          /* if loss exceeds maximum switch off permanently current layer and all higher layers */
          if ((ldd[nl]->percentloss[xx] > cd->maxP100Loss) ||
              (ldd[nl]->percentloss[(xx+cd->considered-1) % cd->considered] > cd->maxP100Loss))
          {
              printf("\nT: %3lu \t\tSWITCHING OFF COMPLETELY LAYER %d (LOSS %d%%).\n\n",
                     curtime, nl, ldd[nl]->percentloss[xx]);
              ldd[nl]->state = THROW_AWAY_ALL;
              for (ii=nl-1;ii>=0;ii--)
              {
                  if (ldd[ii]->state != THROW_AWAY_ALL)
                  {
                      printf("\t\tALSO SWITCHING OFF COMPLETELY LAYER %d.\n\n", ii);
                      ldd[ii]->state = THROW_AWAY_ALL;
                  }
              }
              return HIDE_LAYER;
          }
          break;
      case 12:
          /* if accumulated gob losses since last intra attenuated by a fading factor
             exceed maximum switch off permanently current layer and all higher layers */
          printf("\nT: %3lu Layer: %d | accu-loss: %u", curtime, nl, ldd[nl]->accuPercentLoss);
          if (ldd[nl]->accuPercentLoss > cd->maxP100LossSum)
          {
              printf("\nT: %3lu \t\tSWITCHING OFF COMPLETELY LAYER %d (ACCU. LOSS %d%%).\n\n",
                     curtime, nl, ldd[nl]->accuPercentLoss);
              ldd[nl]->state = THROW_AWAY_ALL;
              for (ii=nl-1;ii>=0;ii--)
              {
                  if (ldd[ii]->state != THROW_AWAY_ALL)
                  {
                      printf("\t\tALSO SWITCHING OFF COMPLETELY LAYER %d.\n\n", ii);
                      ldd[ii]->state = THROW_AWAY_ALL;
                  }
              }
              return HIDE_LAYER;
          }
          break;
      case 13:
          /* if sum of previous 'cd->considered' losses exceeds maximum switch off
             permanently current layer and all higher layers */
          /* hierfuer auch updateLossStatistics, wenn kein einziges Paket fuer Layer ankommt,
             obwohl abgeschickt */
          xx += cd->considered;
          for  (ii=0;ii<cd->considered;ii++)
          {
              xx = (xx - ii) % cd->considered; /* from current to last considered */
              if (ldd[nl]->percentloss[xx] < 0)
                  break;
              addedPercentLoss += ldd[nl]->percentloss[xx];
          }
          if (addedPercentLoss > cd->maxP100LossSum)
          {
              printf("Accumulated gob loss (%d%%) exceeds maximum. LAYER %d NOT DISPLAYED.\n\n",
                     addedPercentLoss, nl);
              return HIDE_LAYER;
          }
          break;
      case 14:
          /* if accumulated gob losses since last intra attenuated by a fading factor 
             exceed maximum switch off permanently current layer and all higher layers */
          printf("\tLayer: %d | MV-loss: %lu.\n", nl, ldd[nl]->mv_loss);
          if (ldd[nl]->mv_loss > 0xefffffff)
          {
              printf("\nT: %3lu \t\tSWITCHING OFF COMPLETELY LAYER %d (ACCU. MB-LOSS %lu%%).\n\n",
                     curtime, nl, ldd[nl]->mv_loss);
              ldd[nl]->state = THROW_AWAY_ALL;
              for (ii=nl-1;ii>=0;ii--)
              {
                  if (ldd[ii]->state != THROW_AWAY_ALL)
                  {
                      printf("\t\tALSO SWITCHING OFF COMPLETELY LAYER %d.\n\n", ii);
                      ldd[ii]->state = THROW_AWAY_ALL;
                  }
              }
              return HIDE_LAYER;
          }
          break;
      case 21:
          /* if current loss exceeds maximum don't display */
          if (ldd[nl]->percentloss[xx] > cd->maxP100Loss)
          {
              printf("Gob loss (%d%%) exceeds maximum. LAYER %d NOT DISPLAYED.\n\n",
                     ldd[nl]->percentloss[xx], nl);
              return HIDE_LAYER;
          }
          break;
      case 22:
          /* if sum of previous 'cd->considered' losses exceeds maximum don't display */
          /* hierfuer auch updateLossStatistics, wenn kein einziges Paket fuer Layer ankommt,
             obwohl abgeschickt */
          xx += cd->considered;
          for  (ii=0;ii<cd->considered;ii++)
          {
              xx = (xx - ii) % cd->considered; /* from current to last considered */
              if (ldd[nl]->percentloss[xx] < 0)
                  break;
              addedPercentLoss += ldd[nl]->percentloss[xx];
          }
          if (addedPercentLoss > cd->maxP100LossSum)
          {
              printf("Accumulated gob loss (%d%%) exceeds maximum. LAYER %d NOT DISPLAYED.\n\n",
                     addedPercentLoss, nl);
              return HIDE_LAYER;
          }
          break;
      case 31:
          /* if loss exceeds maximum switch off current layer and all higher layers
             until next intra */
          if ((ldd[nl]->percentloss[xx] > cd->maxP100Loss) ||
              (ldd[nl]->percentloss[(xx+cd->considered-1) % cd->considered] == 100))
          {
              printf("Gob loss (%d%%) exceeds maximum. LAYER %d NOT DISPLAYED. LOOKING FOR INTRA.\n\n",
                     ldd[nl]->percentloss[xx], nl);
              ldd[nl]->state = LOOK_FOR_INTRA;
              return HIDE_LAYER;
          }
          break;
      default:
          printf("\nWrong display criterium (%d)!\n", cd->displayCriterium);
          exit(1);
    }
    return SHOW_LAYER;
}

/*
 *
 */
int guessMissingPyraFrames(PyraGlobal *pyraData, layerDecData *ldd, concealDispVal *cd,
                           const unsigned long curtime)
{
    unsigned long int deltaT;
    int curFramesMissing, newFramesMissing=0;

    if (pyraData->first)
        return 0;

/* compute deltaT (time since last decoded frame) */
    if (curtime >= ldd->prevDecTime)
        deltaT = curtime - ldd->prevDecTime;
    else
    {
        deltaT = 0xffffffff -(ldd->prevDecTime - curtime - 1); /*!!!!!!!!!!!!! NOCH RICHTIG BERECHNEN */
        assert(0); /*TEST*/
    }
        
    /* compute number of last frames that weren't decoded */
    if (deltaT > pyraData->trd)
    {
        curFramesMissing = ((int)(deltaT / pyraData->trd)) - 1;
        newFramesMissing = curFramesMissing - ldd->prevFramesMissing;
        ldd->prevFramesMissing = curFramesMissing;
        ldd->missFramesSinceI += newFramesMissing;
    }
    else
        ldd->prevFramesMissing = 0;

    return newFramesMissing;
}

/*
 *
 */
int H263DecodeDecision(H263Global *h263Data, layerDecData **ldd, concealDispVal *cd,
                       const unsigned long curtime, int *gotdata, int nl)
{
    unsigned long int deltaT;
    int lastFramesMissing=0;
    layerDecData *cldd = ldd[nl];
    
    switch ( cldd->state )
    {
      case DECODE_ALL:
          if (ShowH263GOBnumber(cldd->slot[cldd->curstart].bp) != 0)
          {
              /* compute deltaT (time since last decoded frame) */
              if (curtime > cldd->prevDecTime)
                  deltaT = curtime - cldd->prevDecTime;
              else
              {
                  deltaT = 0xffffffff -(cldd->prevDecTime - curtime - 1); /*!!!!!!!!!!!!! NOCH RICHTIG BERECHNEN */
                  assert(0); /*TEST*/
              }

              /* compute number of last frames that weren't decoded */
              if (deltaT > h263Data->trd)
              {
                  lastFramesMissing = ((int)(deltaT / h263Data->trd)) - 1;
                  if (cd->ConcealMessages && lastFramesMissing)
                      printf("#### Previous %d frame(s) not decoded!\n\n", lastFramesMissing);
              }

              /* if we aren't sure the picture header hasn't changed we don't decode */
              if (lastFramesMissing > 2)
              {
                  printf("\nPICTURE HEADER MAY HAVE CHANGED IN LAYER %d. LOOKING FOR SYNC.\n\n", nl);
                  cldd->state = LOOK_FOR_PIC_SYNC;
                  throwawayFrame(cldd);
                  UNSETBIT(*gotdata, nl);
              }
              else
              {
                  /* check GobFrameId (compare current + previous) */
                  if (getGFID(cldd) != h263Data->gfid)
                  {
                      printf("\nMISSING PICTURE HEADER HAS CHANGED IN LAYER %d. LOOKING FOR SYNC.\n\n", nl);
                      cldd->state = LOOK_FOR_PIC_SYNC;
                      throwawayFrame(cldd);
                      UNSETBIT(*gotdata, nl);
                  }
              }   
          }          
          break;
      case LOOK_FOR_INTRA:
          if (ShowH263GOBnumber(cldd->slot[cldd->curstart].bp) != 0)
          {
              throwawayFrame(cldd);
              UNSETBIT(*gotdata, nl);
          }
          else
          {
              if (ShowH263CodingType(cldd->slot[cldd->curstart].bp) != PICTURE_CODING_TYPE_INTRA)
              {
                  throwawayFrame(cldd);
                  UNSETBIT(*gotdata, nl);                  
              }
              else
              {
                  printf("\nFOUND INTRA IN LAYER %d. CAN RESYNCHRONIZE.\n\n", nl);
                  cldd->state = DECODE_ALL;
              }
          }
          break;
      case LOOK_FOR_PIC_SYNC:
          if (ShowH263GOBnumber(cldd->slot[cldd->curstart].bp) != 0)
          {
              throwawayFrame(cldd);
              UNSETBIT(*gotdata, nl);
          }
          else
          {
              printf("\nFOUND PICTURE SYNC IN LAYER %d. CAN RESYNCHRONIZE.\n\n", nl);
              cldd->state = DECODE_ALL;
              /*
              if (ShowPyraCodingType(cldd->slot[cldd->curstart].bp) != PICTURE_CODING_TYPE_INTRA)
                  cldd->picCodType = PICTURE_CODING_TYPE_INTRA;
              else
                  cldd->picCodType = PICTURE_CODING_TYPE_INTER;
              */
          }
          break;
      case THROW_AWAY_ALL:
          throwawayFrame(cldd);
          UNSETBIT(*gotdata, nl);
          break;
      default:
          break;
    }
    return 0;
}

/*
 *
 */
int PyraDecodeDecision(PyraGlobal *pyraData, layerDecData **ldd, concealDispVal *cd,
                       const unsigned long curtime, int *gotdata, int nl)
{
    int newFramesMissing=0;
    layerDecData *cldd = ldd[nl];
    
    pyraData->trd = cldd->slot[cldd->curstart].header.prh.tempdist;
    
    switch ( cldd->state )
    {
      case DECODE_ALL:
          /* handle last frames that weren't decoded */
          if ((newFramesMissing = guessMissingPyraFrames(pyraData, cldd, cd, curtime)) > 0)
          {
              updateLossStatistics(pyraData, cldd, cd, COMPLETE_LOSS, newFramesMissing);
              if (cd->ConcealMessages && newFramesMissing)
                  printf("#### Previous %d frame(s) not decoded!\n\n", newFramesMissing);
          }

          /* do we have a picture sync? */
          if (ShowPyraGOBnumber(cldd->slot[cldd->curstart].bp) != 0)
          {              
              /* if we aren't sure the picture header hasn't changed we don't decode */
              if (cldd->prevFramesMissing > 2)
              {
                  printf("\nPICTURE HEADER MAY HAVE CHANGED IN LAYER %d. LOOKING FOR SYNC.\n\n", nl);
                  cldd->state = LOOK_FOR_PIC_SYNC;
                  throwawayFrame(cldd);
                  UNSETBIT(*gotdata, nl);
              }
              else
              {
                  /* check GobFrameId (compare current + previous) */
                  if (getGFID(cldd) != pyraData->GOBFrameID)
                  {
                      printf("\nMISSING PICTURE HEADER HAS CHANGED IN LAYER %d. LOOKING FOR SYNC.\n\n", nl);
                      cldd->state = LOOK_FOR_PIC_SYNC;
                      throwawayFrame(cldd);
                      UNSETBIT(*gotdata, nl);
                  }
              }
          }
          else if (ShowPyraCodingType(cldd->slot[cldd->curstart].bp) == PICTURE_CODING_TYPE_INTRA)
          {
              cldd->picCodType = PICTURE_CODING_TYPE_INTRA;                        
          }
          else
              cldd->picCodType = PICTURE_CODING_TYPE_INTER;
          break;
      case LOOK_FOR_INTRA:
          if (ShowPyraGOBnumber(cldd->slot[cldd->curstart].bp) != 0)
          {
              throwawayFrame(cldd);
              UNSETBIT(*gotdata, nl);
          }
          else
          {
              if (ShowPyraCodingType(cldd->slot[cldd->curstart].bp) != PICTURE_CODING_TYPE_INTRA)
              {
                  cldd->picCodType = PICTURE_CODING_TYPE_INTER;                        
                  throwawayFrame(cldd);
                  UNSETBIT(*gotdata, nl);                  
              }
              else
              {
                  printf("\nFOUND INTRA IN LAYER %d. CAN RESYNCHRONIZE.\n\n", nl);
                  cldd->picCodType = PICTURE_CODING_TYPE_INTRA;                        
                  cldd->state = DECODE_ALL;
              }
          }
          break;
      case LOOK_FOR_PIC_SYNC:
          /* handle last frames that weren't decoded */
          if ((newFramesMissing = guessMissingPyraFrames(pyraData, cldd, cd, curtime)) > 0)
          {
              updateLossStatistics(pyraData, cldd, cd, COMPLETE_LOSS, newFramesMissing);
              if (cd->ConcealMessages && newFramesMissing)
                  printf("#### Previous %d frame(s) not decoded!\n\n", newFramesMissing);
          }

          if (ShowPyraGOBnumber(cldd->slot[cldd->curstart].bp) != 0)
          {
              throwawayFrame(cldd);
              UNSETBIT(*gotdata, nl);
          }
          else
          {
              printf("\nFOUND PICTURE SYNC IN LAYER %d. CAN RESYNCHRONIZE.\n\n", nl);
              cldd->state = DECODE_ALL;              
              if (ShowPyraCodingType(cldd->slot[cldd->curstart].bp) != PICTURE_CODING_TYPE_INTRA)
                  cldd->picCodType = PICTURE_CODING_TYPE_INTRA;
              else
                  cldd->picCodType = PICTURE_CODING_TYPE_INTER;
          }
          break;
      case THROW_AWAY_ALL:
          throwawayFrame(cldd);
          UNSETBIT(*gotdata, nl);
          break;
      default:
          break;
    }
    return 0;
}

/*
 *
 */
int getGFID(layerDecData *ldd)
{
    int xx, cont=1, gfid=4; /* invalid value for gfid */
    unsigned char *bp;
    unsigned long int firstlong;

    xx = ldd->curstart;
    if (ldd->layertype == LAYERTYPE_H263)
    {
        while (cont)
        {
            if (xx == ldd->curend) cont = 0;
            if ( !ldd->slot_f[xx] )
            {
                xx = (xx + 1) % B_SLOTS;
                continue;
            }
            bp = ldd->slot[xx].bp;
            firstlong = (bp[0]<<24)+(bp[1]<<16)+(bp[2]<<8)+bp[3];
            if ((firstlong & 0xffff8000) == 0x00008000)
            {
                gfid = ShowH263GFID(bp);
                break;
            }
            xx = (xx + 1) % B_SLOTS;
        }
    }
    else
    {
        while (cont)
        {
            if (xx == ldd->curend) cont = 0;
            if ( !ldd->slot_f[xx] )
            {
                xx = (xx + 1) % B_SLOTS;
                continue;
            }
            bp = ldd->slot[xx].bp;
            firstlong = (bp[0]<<24)+(bp[1]<<16)+(bp[2]<<8)+bp[3];
            if ((firstlong & 0xffffff00) != 0x00000100) 
            {
                gfid = ShowPyraGFID(bp);
                break;
            }
            xx = (xx + 1) % B_SLOTS;
        }        
    }
    ldd->gfid = gfid;
    return gfid;
}
