/*
 * chop.c -- a general purpose sync word chopper
 *
 * 26.2.98, jw
 */
#include <stdio.h>
#include <jittr/dstring.h>

#ifdef DEBUG
FILE *debugfp = stderr;
#endif

struct PacketHdr
{
  unsigned char seqno;		/* counts pictures */
  unsigned char nextseq;	/* seqno of next packet */
  unsigned char frag; 		/* this packet starts with the frag'th fragment 
  				 * of picture seqno; frag counts from 0 on. 
				 */
  unsigned int payload;		/* nr. of valid data bytes after header */
};

#define HDR_BYTES 6		/* encoded size of a PacketHdr */

struct StreamInfo
{
  FILE *fp;
  unsigned char *pat;
  int pbits;
  int pbytes;
  int middle_sync;	/* offset of a sync word not at start/end of dstring */
};

/*
 * ChopByteSync finds the first nbits bits of pat in the file referenced by fp.
 * Start of the sync pattern pat is expected to be byte aligned.
 * all material from the current file position up to and including the bytes
 * containing pat are appended to *d.
 * to read an unlimited amount of bytes use 0 for max.
 *
 * returns nonzero at EOF.
 */
int ChopByteSync(dstring **d, FILE *fp, int max, unsigned char *pat, int nbits)
{
  unsigned char *s, *p = pat;
  int c, need = nbits;
  int nbytes = (nbits + 7) >> 3;

  if (!*d) dstring_append(d, 0, NULL, 8192);

  if (max) max++;

  for (;;)
    {
      if (((c = getc(fp)) == EOF) || !--max)
	{
	  (*d)->buf[(*d)->length] = '\0';
	  return -1;
	}
      if ((*d)->allocated <= (*d)->length) dstring_append(d, -1, NULL, 8192);
      *(s = (*d)->buf + (*d)->length++) = c;

      if ((*d)->length < nbytes)
        continue;
	
      s += 1-nbytes;
      
      while (need >= 8)
        {
	  if (*s == *p)
	    {
	      s++; p++;
	      need -= 8;
	    }
	  else
	    break;
	}
      if (!need || ((need < 8) && ((*s >> (8-need)) == (*p >> (8-need)))))
        break;
      p = pat;
      need = nbits; 
    }
  (*d)->buf[(*d)->length] = '\0';
  return 0;
}

int WriteHdr(unsigned char *b, struct PacketHdr *h, int s, int n, int f, int l)
{
  debug2("WriteHdr: s=%d, n=%d, ", s, n);
  debug2("f=%d, l = %d\n", f, l);
  h->seqno = s;
  h->nextseq = n;
  h->frag = f;
  *b++ = s;
  *b++ = n;
  *b++ = f;
  *b++ = (l >> 16) & 0xff;
  *b++ = (l >> 8) & 0xff;
  *b++ = l & 0xff;
  ASSERT(HDR_BYTES == 6);
  ASSERT(!(l >> 24));
  return 0;
}

struct StreamType
{
  unsigned char *name;
  unsigned char *sync;
  int sbits;
} 
stypes[] =
{
  { "ScalVico Pyra", "\0\0\1\0", 8+8+8+5 },
  { "H.263",         "\0\0\200", 8+8+6   },
  { "Audio MPEG L2", "\377\375\200", 8+8+4 },
  { NULL, NULL, 0 }
};

int main(int ac, char ** av)
{
  FILE *ifp = fopen(av[1], "r");
  int i = 0;
  dstring *d = NULL;
  dstring *packet = NULL;
  struct StreamType *t;
  int pbytes;
  struct PacketHdr phdr;
  struct StreamInfo s;
  
  if (!ifp) perror(av[1]);

  for (t = stypes; t->sync; t++)
    {
      if (!ChopByteSync(&d, ifp, (t->sbits+7) >> 3, t->sync, t->sbits))
        break;
      d->length = 0;
      rewind(ifp);
    }
 
  if (!t->sync)
    {
      fprintf(stderr, "hmm, %s does no start with a sync word of ", av[0]);
      for (t = stypes; t->sync; t++)
	fprintf(stderr, "%s%s", (t == stypes) ? "" : ", ", t->name);
      fprintf(stderr, "\n");
      return -1;
    }
  fprintf(stderr, "Stream Type: %s\n", t->name);
    
  pbytes = d->length;

  /* ChopByteSync includes the next sync word in d, when returning 0 */
  while (!ChopByteSync(&d, ifp, 0, t->sync, t->sbits))
    {
      i++;
      d->length -= pbytes;
      /* ignore the last sync word now */
      printf("%06x: %d: %d bytes\n", ftell(ifp), i, d->length);
      /* move the remaining sync word to the beginning of the buffer */
      xbcopy(d->buf + d->length, d->buf, pbytes);
      d->length = pbytes;
    }
  /* no more picture sync words, but possibly body of last picture */
  printf("%06x: %d: %d bytes\n", ftell(ifp), ++i, d->length);
}
