/*
 * jpumpd.c -- a jittr daemon that can feed video-streams via udp
 *
 * jw, 7.3.98, setup and play a la rtsp.
 */

#include <stdio.h>              /* NULL */
#include <unistd.h>		/* for close() */
#include <errno.h>		/* for strerror(errno) */

#include <sys/types.h>          /* for sys/socket.h */
#include <sys/socket.h>         /* SOCK_STREAM */
#include <sys/time.h>           /* struct timeval for net.h */

#include <jittr/atom.h>		/* struct dstring */
#include <jittr/parse.h>	/* jarg_*() methods */
#include <jittr/jittr.h>	/* jittr_error(), jittr_arg[cv] */
#include <jittr/net.h>		/* PT_DISABLED */
#include <jittr/doio.h>		/* TurnClient() */
#include <jittr/schedule.h>	/* sched_enter(), sched_alloc() */

#include "pump.h"		/* for human consumption only */

enum a_idx {
  SETUP, STATUS, PLAY, STOP, TEARDOWN,
  SIZEOF_A_IDX
};

struct pump
{
  int fd;		/* device fildedescriptor */
  struct client *cl;
  int poll_id;
  int poll_timeout;	/* cleared by pump_interpret, set by pump_poll */
  int a[SIZEOF_A_IDX];
  enum a_idx *ai;
  int ai_min, ai_max;
  char lastLEDs[10];
#define IS_V_AI(v, n) (((n) >= (v)->ai_min) && ((n) <= (v)->ai_max))
#define V_AI(v, n) ((v)->ai[(n) - (v)->ai_min])
};

static int pump_trace __P((int idx, int flag, void *data));
static int pump_input __P((void *cli, char *bufp, int len));
static int pump_interpret __P((struct pump *v, unsigned char *buf, int l));
static int pump_poll __P((void **av));

char jpumpd_modname[] = "pumpd v0.1 server a file via udp";

char *jpumpd_modhelp[] = 
{
  /* name		description */
  ".",			"start the daemon",
  NULL
};


/*
 * A simple syntax check is to look for /"[^,][ 	]
 */
char *jpumpd_modstruct[] =
{
#define OFF(e)  (char *)(&((struct pump *)0)->a[e])
/*off   name                            type	value	*/
  /* -- ------------------------------  ------  ------- */
  OFF(HEXCMD),	"./hexcmd",		"a",	"",		/* better don't touch  */
  OFF(STATUS),	"./status",		"a",	"init",		/* see what is happening */
  OFF(INPUT),	"./input",		"a",	"fbas",
  (char *)-1,	"./input/.state",	"v",	"{FBAS YC}",
  OFF(FREEZE),	"./freeze",		"a",	"off",
  (char *)-1,	"./freeze/.state",	"v",	"{off on}",
  OFF(TYPE),	"./type",		"a",	"unknown",
  (char *)-1,	"./type/.state",	"v",	"{pal ntsc unknown}",
  OFF(COLOR),	"./color",		"a",	"64",
  (char *)-1,	"./color/.min",		"v",	"1",
  (char *)-1,	"./color/.max",		"v",	"127",
  OFF(TINT),	"./tint",		"a",	"64",
  (char *)-1,	"./tint/.min",		"v",	"1",
  (char *)-1,	"./tint/.max",		"v",	"127",
  OFF(CONTRAST),"./contrast",		"a",	"64",
  (char *)-1,	"./contrast/.min",	"v",	"1",
  (char *)-1,	"./contrast/.max",	"v",	"127",
  OFF(PHASE),	"./phase",		"a",	"32",
  (char *)-1,	"./phase/.min",		"v",	"1",
  (char *)-1,	"./phase/.max",		"v",	"63",

  NULL,		NULL
};

int 
jpumpd_modmain(atom_index, flag, private_data)
int atom_index, flag;
void **private_data;
{
  int i, l;
  char *port = JITTR_DEFAULT_TTY;
  struct pump *v = NULL;

  if (!*private_data && (flag & TRACE_WRITE))
    {
      for (i = 0; i < jittr_argc; i++)
        {
	  if ((l = strlen(jittr_argv[i])) < 2)
	    continue;
	  if (!strncmp(jittr_argv[i], "-device", l))
	    {
	      jittr_shift_argv(i);
	      if (!(port = jittr_argv[i]) || !*port)
	        return jittr_error("%s: option -dev: parameter missing.\n",
			jpumpd_modname, 0, 0, 0);
	      jittr_shift_argv(i--);
	    }
	  else if (!strncmp(jittr_argv[i], "-name", l))
	    {
	      jittr_shift_argv(i);
	      if (jittr_change_name(atom_index, jittr_argv[i]))
	        return jittr_error("%s: option -name: bad parameter.\n",
			jpumpd_modname, 0, 0, 0);
	      jittr_shift_argv(i--);
	    }
	  else if (!strncmp(jittr_argv[i], "-help", l))
	    {
	      jittr_error("%s:\n  -dev <serial-line>\t\tDefault: %s\n",
			  jpumpd_modname, port, 0, 0);
	      jittr_error("  -name <logical-name>\t\tDefault: %s\n",
	      		  atom_name(atom_index), 0, 0, 0);
	      return 0;
	    }
	}
      *private_data = NULL;
    }

  if (!*private_data && (flag & TRACE_EXEC))
    {
      if (!(port = jarg_first_word(NULL)))
	port = JITTR_DEFAULT_TTY;
      debug2("running pumpd after modload on atom %d. device=%s.\n",
      	atom_index, port);
    }

  if (*private_data)
    return jittr_error("%s already initialized on atom %d.\n", 
      		jpumpd_modname, atom_index, 0, 0);
  
  if (!(v = (struct pump *)calloc(1, sizeof(struct pump))))
    return jittr_error("%s: %d: no memory.\n", jpumpd_modname, atom_index,0,0);
  *private_data = (void *)v;		/* our modexit() will free it */

  if (!(v->cl = jittr_init_tty(atom_index, port, DEFAULT_SPEED)))
    return jittr_error("%s: atom %d: tty_init %s failed: %s.\n",
    		jpumpd_modname, atom_index, port, strerror(errno));

  v->cl->input_notify_fn = pump_input;
  v->cl->input_notify_data = *private_data;

  /* create an instance of the devices world */
  if (jittr_mount(atom_index, jpumpd_modstruct, (void *)v))
    return jittr_error("%s: atom %d: init failed.\n", 
	    jpumpd_modname, atom_index, 0, 0);
   
  v->ai_min = 1000000;
  v->ai_max = 0;
  for (i = 0; i < SIZEOF_A_IDX; i++)
    {
      atom_trace_set(v->a[i], TRACE_RDWR|TRACE_EXEC, pump_trace, (void *)v);
      if (v->a[i] > v->ai_max) v->ai_max = v->a[i];
      if (v->a[i] < v->ai_min) v->ai_min = v->a[i];
#ifdef DEBUG
      if (v->a[i] <= 0)
        debug2("%s: enum v->a[%d] not initialized?\n", jpumpd_modname, i);
#endif
    }
  
  /* construct the reverse index */
  v->ai = (enum a_idx *)calloc(sizeof(enum a_idx), v->ai_max + 1 - v->ai_min);
  for (i = 0; i < SIZEOF_A_IDX; i++)
    V_AI(v, v->a[i]) = i;

  v->poll_id = sched_enter(sched_alloc(gettimeval, 10, 0, pump_poll,(void *)v,0));
  return 0;
}

/*
 * modexit() is responsible to remove all traces that have been set on any
 * atoms by this instance of the module
 */
int
jpumpd_modexit(atom_index, private_data)
int atom_index;
void **private_data;
{
  struct pump *v = (struct pump *)*private_data;

  if (v)
    {    
      /*
       * remove hierarchy, kill, close things, ...
       */
      v->cl->input_notify_data = NULL;  /* tell pump_input() what happened */
      TurnClient(ClientByFd(NULL, v->cl->i->fd), 1);
      jittr_umount(atom_index);
      close(v->fd);	/* just to be sure. Should be done by TurnClient() */

      debug2("modexit done '%s' atom %d.\n", jpumpd_modname, atom_index);
      free((char *)*private_data); *private_data = NULL;
    }

  return 0;
}

static int
pump_input(cli, bufp, len)
void *cli;
char *bufp;
int len;
{
  struct client *cl = (struct client *)cli;
  struct pump *v = (struct pump *)cl->input_notify_data;
  struct dstring *buf;
  int p = 0, l, i, r = 0;

  if (!bufp || !len)
    {
      debug("empty pump read?\n");
      return 0;           	/* considererd o.k. */
    }
  buf = cl->i->ibuf;
  ASSERT(bufp + len == buf->buf + buf->length);
      
#ifdef DEBUG
  {
    static struct dstring *dbgd = NULL;
    jittr_hexdump(&dbgd, bufp, len, "pump: ");
    debug1("%s", dbgd->buf);
    dbgd->length = 0;
  }
#endif

  for (i = 0; i < buf->length; i++)
    {
      p = i;
      for (l = i; l < buf->length; l++)
	{
	  if (((unsigned char)buf->buf[l] == '\n') ||
	      ((unsigned char)buf->buf[l] == '\r') ||
	      ((unsigned char)buf->buf[l] == ' '))
	    {
	      /* end of packet seen */
	      p = l + 1;
	      r |= pump_interpret(v, (unsigned char *)buf->buf+i, l-i);
	      i = l;
	      break;
	    }
	}
    }
  if (p)
    xbcopy(buf->buf + p, buf->buf, buf->length - p);
  buf->length -= p;
  return r;
}

/*
 * pump_interpret expects complete single packets in buf.
 * It updates the status word and evaluates jittr commands 
 * according to the changes.
 */
static int
pump_interpret(v, buf, len)
struct pump *v;
unsigned char *buf;
int len;
{
  struct timeval tv;
  static inside_pump_interpret = 0;

  if (len < 1)
    return 0;
  
  if (inside_pump_interpret)
    {
      debug2("recursive call to pump_interpret('%s', %d);\n", buf, len);
      return 0;
    }
  inside_pump_interpret++;

  v->poll_timeout = 0;	/* set by pump_poll() */

  while (len && (buf[len-1] > '9' || buf[len-1] < '0'))
    len--;

  jittr_gettimeval(&tv);
       if (!strncmp(buf, "Col", 3)) 
    jittr_atom_set(v->a[COLOR],    0, buf+3, len-3, &tv, v->cl, 0);
  else if (!strncmp(buf, "Con", 3))
    jittr_atom_set(v->a[CONTRAST], 0, buf+3, len-3, &tv, v->cl, 0);
  else if (!strncmp(buf, "C",   1))
    {
      char *p = (buf[1] == '2') ? "1" : "0";
      jittr_atom_set(v->a[INPUT], 0, p, 1, &tv, v->cl, 0);
    }
  else if (!strncmp(buf, "Tin", 3))
    jittr_atom_set(v->a[TINT], 0, buf+3, len-3, &tv, v->cl, 0);
  else if (!strncmp(buf, "T",   1))
    {
      char *p = (buf[1]=='2') ? "pal" : ((buf[1]=='1') ? "ntsc" : "unknown");
      jittr_atom_set(v->a[TYPE], 0, p, strlen(p), &tv, v->cl, 0);
    }
  else if (!strncmp(buf, "Hph", 3))
    jittr_atom_set(v->a[PHASE], 0, buf+3, len-3, &tv, v->cl, 0);
  else if (!strncmp(buf, "Frz", 3))
    jittr_atom_set(v->a[FREEZE], 0, buf+3, len-3, &tv, v->cl, 0);
  else if (!strncmp(buf, "E01", 3)) 
    jittr_atom_set(v->a[STATUS], 0, "E01: channel does not exist", 27, &tv, v->cl, 0);
  else if (!strncmp(buf, "E06", 3)) 
    jittr_atom_set(v->a[STATUS], 0, "E06: dip switch 'AUTO SW' wrong", 31, &tv, v->cl, 0);
  else if (!strncmp(buf, "E13", 3)) 
    jittr_atom_set(v->a[STATUS], 0, "E13: parameter out of range", 27, &tv, v->cl, 0);
  else if (!strncmp(buf, "RECONFIG", 8)) 
    {
      void *av[2]; 

      av[1] = NULL; av[0] = (void *)v; 
      pump_poll(av);
    }
  else
    jittr_atom_set(v->a[STATUS], 0, buf, len, &tv, v->cl, 0);

  inside_pump_interpret--;

  return CIN_SCHED_MASK;
}

static int
pump_trace(idx, flag, data)
int idx;
int flag;
void *data;
{
  struct pump *v = (struct pump *)data;
  struct dstring **o = &v->cl->o->obuf;
  struct atom_data *a = jittr_atom_data(idx);
  struct client *who = a->who;
  struct timeval tv;
  char *s;
  int i, num;

  tv.tv_sec = a->sec; tv.tv_usec = a->usec;

  if (!IS_V_AI(v, idx))
    return jittr_error("pump_trace: Do not know what to do with atom %s.\n",
		  atom_name(idx), 0, 0, 0);

  if (flag & TRACE_EXEC)
    {
      s = jarg_first_word(&i);
      if (exec_context)
        who = exec_context->who;
      jittr_gettimeval(&tv);
      switch (V_AI(v, idx))
        {
	case HEXCMD:
	  jittr_atom_set(idx, 0, s, i, &tv, who, 1);
	  break;
	default:
	  jittr_atom_set(idx, 0, s ? s : "exec", s ? i : 4, &tv, who, 1);
	}
      return 0;
    }

  if (flag & TRACE_READ)
    {
      /* if its a get from a trace -w callback, a flag is set in the atom */
      if (!atoms[idx]->a.traced & TRACE_WRITE)
	switch (V_AI(v, idx))
	  {
	  default:
	    return jittr_error("pump_trace: %s: nothing happens.\n",
	    	atom_name(idx), 0, 0, 0);
	  }
      return 0;
    }

  ASSERT(flag & TRACE_WRITE);

  if ((i = jittr_check_numeric(idx, &num, NULL)) < 0)
    return -1;	/* numeric type detected, but desasterous value */
  else if (i > 0)
    num = -1;	/* atom has not a numeric type */

  switch (V_AI(v, idx))
    {
    case HEXCMD:	
      dstring_append(o, -1,  atom_value(idx), atoms[idx]->value->length);
      jittr_error("pump_trace: %d bytes of raw data sent!\n", 
      		  atoms[idx]->value->length, 0, 0, 0);
      break;
    case STATUS:
    case TYPE:
      break;	/* ignored */
    
    case INPUT:    dstring_appendn(o, -1, "%d!\ni\n", num + 1);     break;
    case FREEZE:   dstring_appendn(o, -1, "%c\n",  num?'|':'~'); break;
    case COLOR:    dstring_appendn(o, -1, "%dC\n", num); break;
    case TINT:     dstring_appendn(o, -1, "%dT\n", num); break;
    case CONTRAST: dstring_appendn(o, -1, "%d^\n", num); break;
    case PHASE:    dstring_appendn(o, -1, "%dH\n", num); break;

    default:
      debug1("pump_trace for %s not impl.\n", atom_name(idx));
      break;
    }
  return 0;
}


/*
 * pump_poll checks the activity of the device.
 */
static int
pump_poll(av)
void **av;
{
  struct pump *v = (struct pump *)av[0];
  struct dstring **o = &v->cl->o->obuf;
  static dstring *d = NULL;
  struct timeval tv;

  dstring_append(o, -1, "i\n", 2);
  if (v->poll_timeout)
    {
      jittr_gettimeval(&tv);
      debug1("poll_timeout %d\n", v->poll_timeout);
      dstring_append(&d, 0, "? no connection to device", 0);
      jittr_atom_set(v->a[STATUS], 0, d->buf, d->length, &tv, v->cl, 0);
    }
  else if (*atom_value(v->a[STATUS]) == '?')
    {
      jittr_gettimeval(&tv);
      dstring_append(&d, 0, "init", 4);
      jittr_atom_set(v->a[STATUS], 0, d->buf, d->length, &tv, v->cl, 0);
    }
  v->poll_timeout++;

  return 0;
}
