/*
 * pack.c -- a general purpose sync word chopper and packet assembler
 *
 * 28.2.98, jw
 */
#include <stdio.h>
#include <jittr/dstring.h>

#ifdef DEBUG
FILE *debugfp = stderr;
#endif

struct PacketHdr
{
  int seqno;		/* counts packets */
  int payload;		/* nr. of valid data bytes after header */
};

#define HDR_BYTES 4		/* encoded size of a PacketHdr */

struct StreamInfo
{
  FILE *fp;
  unsigned char *pat;
  int pbits;
  int pbytes;
  int picend_seen;		/* does the temporary buffer finish a pic? */
};

/*
 * ChopByteSync finds the first nbits bits of pat in the file referenced by fp.
 * Start of the sync pattern pat is expected to be byte aligned.
 * all material from the current file position up to and including the bytes
 * containing pat are appended to *d.
 * to read an unlimited amount of bytes use 0 for max.
 *
 * returns nonzero at EOF.
 */
int ChopByteSync(dstring **d, FILE *fp, int max, unsigned char *pat, int nbits)
{
  unsigned char *s, *p = pat;
  int c, need = nbits;
  int nbytes = (nbits + 7) >> 3;

  if (!*d) dstring_append(d, 0, NULL, 8192);

  if (max) max++;

  for (;;)
    {
      if (((c = getc(fp)) == EOF) || !--max)
	{
	  (*d)->buf[(*d)->length] = '\0';
	  return -1;
	}
      if ((*d)->allocated <= (*d)->length) dstring_append(d, -1, NULL, 8192);
      *(s = (*d)->buf + (*d)->length++) = c;

      if ((*d)->length < nbytes)
        continue;
	
      s += 1-nbytes;
      
      while (need >= 8)
        {
	  if (*s == *p)
	    {
	      s++; p++;
	      need -= 8;
	    }
	  else
	    break;
	}
      if (!need || ((need < 8) && ((*s >> (8-need)) == (*p >> (8-need)))))
        break;
      p = pat;
      need = nbits; 
    }
  (*d)->buf[(*d)->length] = '\0';
  return 0;
}

int WriteHdr(unsigned char *b, struct PacketHdr *h, int s, int l)
{
  debug2("WriteHdr: s=%d, l=%d, ", s, l);
  h->seqno = s;
  *b++ = s;
  *b++ = (l >> 16) & 0xff;
  *b++ = (l >> 8) & 0xff;
  *b++ = l & 0xff;
  ASSERT(HDR_BYTES == 4);
  ASSERT(!(l >> 24));
  return 0;
}

/*
 * MakePacket returns the number of pictures (as chopped at a sync pattern)
 * that are completed by this packet. This number is identical to the nr. of
 * frame durations, that the transmitter should wait before sending the next
 * packet. A return value of 0 means that the only picture occupying this
 * packet is longer than the maximum packet size. more fragments will follow in
 * further packets. 
 * A negative return value means, that the datastream has ended and the
 * transmitter should not call MakePacket again. No packet has been
 * constructed.
 */
int MakePacket(dstring **p, struct PacketHdr *h, int min_s, int max_s, dstring **tmp)
{
  dstring *d = *tmp;
  struct StreamInfo *s = (struct StreamInfo *)d->data;
  int c = 0, r = 0, l, ll;
  
  if (feof(s->fp) && !d->length)
    {
      /* stream ended */
      (*p)->length = 0;
      return -1;
    }

  dstring_append(p, 0, NULL, ll = HDR_BYTES); 
  (*p)->length = ll;	/* spare room for header */
  dstring_append(p, -1, d->buf, d->length);
  r = s->picend_seen;
  s->picend_seen = 0;

  while (!c && ((*p)->length < (max_s + s->pbytes)))
    { 
      ll = (*p)->length;
      if (ll > HDR_BYTES) ll -= s->pbytes;
      c = ChopByteSync(p, s->fp, 0, s->pat, s->pbits);
      r++;
    }

  l = ((*p)->length < max_s) ? (*p)->length : max_s;
    
  /* 
   * ll now points behind the last picture entirely inside the buffer
   * If ll is above min_s, we cut there. otherwise we cut the picture 
   * exactly at the max_s. The rest is stored in *tmp. We expect to get passed
   * tmp next time unchanged.
   */
  if ((ll < min_s) || ((*p)->length <= max_s))
    ll = l;
  if (((*p)->length - ll) > s->pbytes)
    {
      /* one picture end is not shipped this time */
      s->picend_seen = 1;
      r--;
    }
  dstring_append(tmp, 0, (*p)->buf + ll, (*p)->length - ll);
  (*p)->length = ll;
  WriteHdr((*p)->buf, h, h->seqno + 1, (*p)->length - HDR_BYTES);
  return r;
}

struct StreamType
{
  unsigned char *name;
  unsigned char *sync;	/* what a sync word looks like */
  int sync_bits;	/* how long a sync word is */
  int tref_offset;	/* where a tempref is. Measured from start of sync */
  int tref_bits;	/* how long a tempref is */
} 
stypes[] =
{
  { "ScalVico Pyra", "\0\0\1\0",     8+8+8+5, 0, 0 },
  { "H.263",         "\0\0\200",     8+8+6,   0, 0 },
  { "Audio MPEG L2", "\377\375\200", 8+8+4,   0, 0 },
  { NULL,	     NULL,	     0,       0, 0 }
};


int main(int ac, char ** av)
{
  FILE *ifp = fopen(av[1], "r");
  int i = 0;
  dstring *d = NULL;
  dstring *packet = NULL;
  struct StreamType *t;
  int pbytes;
  struct PacketHdr phdr;
  struct StreamInfo s;
  
  if (!ifp) perror(av[1]);

  for (t = stypes; t->sync; t++)
    {
      if (!ChopByteSync(&d, ifp, (t->sbits+7) >> 3, t->sync, t->sbits))
        break;
      d->length = 0;
      rewind(ifp);
    }
 
  if (!t->sync)
    {
      fprintf(stderr, "hmm, %s does no start with a sync word of ", av[0]);
      for (t = stypes; t->sync; t++)
	fprintf(stderr, "%s%s", (t == stypes) ? "" : ", ", t->name);
      fprintf(stderr, "\n");
      return -1;
    }
  fprintf(stderr, "Stream Type: %s\n", t->name);
    
  pbytes = d->length;

  d->data = (void *)&s;
  s.fp = ifp;
  s.pat = t->sync;
  s.pbits = t->sbits;
  s.pbytes = pbytes;
  s.picend_seen = 0;
  
  phdr.seqno = -1;
  
  for (;;)
    {
      i = MakePacket(&packet, &phdr, 4*512, 4*1024, &d);
      fprintf(stderr, "MakePacket=%d, dlen=%d, des=%d\n", i, d->length,
        s.picend_seen);
      if (i < 0)
        break;
      write(1, packet->buf + HDR_BYTES, packet->length - HDR_BYTES);
    }
  return 0;
}
