/*
 * Copyright (c) 1993-1994 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and the Network Research Group at
 *      Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
static const char rcsid[] =
    "@(#) $Header: /src/multimedia/LOCAL/vic/vic-Marcus/RCS/decoder-h261.cc,v 1.5 1998/03/06 15:47:44 msmeissn Exp msmeissn $ (LBL)";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
#include "decoder.h"
#include "p64.h"
#include "module.h"
#include "renderer.h"

#define FLAG_GOB(flag)		(((flag) >> 20)&0x0f)
#define FLAG_MBA(flag)		(((flag) >> 15)&0x1f)
#define FLAG_MQUANT(flag)	(((flag) >> 10)&0x1f)
#define FLAG_SBIT(flag)		(((flag) >> 29)&0x7)
#define FLAG_EBIT(flag)		(((flag) >> 26)&0x7)

class H261Decoder : public Decoder {
public:
	H261Decoder();
	~H261Decoder();
	int colorhist(u_int* hist) const;
	virtual void recv(const rtphdr* rh, const u_char* bp, int cc);
	virtual void info(char* wrk) const;
	void stats(char* wrk);

protected:
	int reassemble(const u_char*bp,int cc,const rtphdr* rh,u_int *h261streamsize);
	virtual void redraw();

	u_short seqno_;		/* last rtp seqno */
	/*
	 * Reassembly buffer 'slots' for resequencing & reassembling
	 * gobs split across packet boundaries.
	 */
#define H261_SLOTS	64
#define H261_SLOTMASK	(H261_SLOTS - 1)
	int	h261streamsize_;	/* max streamsize */
	u_char	*h261stream_;		/* h261stream */

	struct slot {
		u_short seqno;
		u_int	rtflags;
		u_int	h261header;
		u_int	cc;
		u_char* bp;
		int pad;
	} slot_[H261_SLOTS];
	
	void decode(const u_char* vh, const u_char* bp, int cc);
	P64Decoder* codec_;
	int h261_rtp_bug_;
};

static class H261DecoderMatcher : public Matcher {
public:
	H261DecoderMatcher() : Matcher("decoder") {}
	TclObject* match(const char* id) {
//		fprintf(stderr,"H261DecoderMatcher::match(%s)\n",id);
		if (strcasecmp(id, "h261") == 0)
			return (new H261Decoder());
		else
			return (0);
	}
} dm_h261;

#define STAT_BAD_PSC	0
#define STAT_BAD_GOB	1
#define STAT_BAD_SYNTAX	2
#define STAT_BAD_FMT	3
#define STAT_BAD_SIZE	4
#define STAT_FMT_CHANGE 5
#define STAT_BAD_HEADER 6

H261Decoder::H261Decoder() : Decoder(4), codec_(0), h261_rtp_bug_(0)
{
	stat_[STAT_BAD_PSC].name = "H261-Bad-PSC";
	stat_[STAT_BAD_GOB].name = "H261-Bad-GOB";
	stat_[STAT_BAD_SYNTAX].name = "H261-Bad-Syntax";
	stat_[STAT_BAD_FMT].name = "H261-Bad-fmt";
	stat_[STAT_BAD_SIZE].name = "H261-Bad-Size";
	stat_[STAT_BAD_HEADER].name = "H261-Bad-Header";
	stat_[STAT_FMT_CHANGE].name = "H261-Fmt-change";
	nstat_ = 7;
	decimation_ = 411;

	/* cause a resize trigger later */
	inw_ = 0;
	inh_ = 0;

	seqno_ = 0;
	h261streamsize_ = 0;
	h261stream_ = new u_char[1];
	for (int i = 0; i < H261_SLOTS; ++i) {
		slot_[i].cc = 0;
		slot_[i].seqno = 0;
		slot_[i].rtflags = 0;
		slot_[i].h261header = 0;
		slot_[i].bp = new u_char[2000];
		memset(slot_[i].bp,0,2000);
	}
}

H261Decoder::~H261Decoder()
{
	if (codec_) delete codec_;
	for (int i = 0; i < H261_SLOTS; ++i)
		delete[] slot_[i].bp;
	delete[] h261stream_;
}

void H261Decoder::info(char* wrk) const
{
	if (codec_ == 0)
		*wrk = 0;
	else
		sprintf(wrk, "[q=%d]", codec_->gobquant());
}

void H261Decoder::stats(char* wrk)
{
	/* pull stats out of vic indepdendent P64Decoder */
	setstat(STAT_BAD_PSC, codec_->bad_psc());
	setstat(STAT_BAD_GOB, codec_->bad_GOBno());
	setstat(STAT_BAD_SYNTAX, codec_->bad_bits());
	setstat(STAT_BAD_FMT, codec_->bad_fmt());
	Decoder::stats(wrk);
}

int H261Decoder::colorhist(u_int* hist) const
{
	const u_char* frm = codec_->frame();
	int w = inw_;
	int h = inh_;
	int s = w * h;
	colorhist_411_556(hist, frm, frm + s, frm + s + (s >> 2), w, h);
	return (1);
}


/*
 * Packets waiting for reassembly are kept in 'slots'
 * indexed by the low bits of the sequence number.
 * Packets have different sizes for they are split on MB borders.
 * this makes reassembly a bit messy. *sigh*
 *
 * the resulting stream should also start at a zero border while the packets
 * may start at any border (well, the ones generated by the grabber-xil do.
 */
inline int
H261Decoder::reassemble(const u_char* bp, int cc,const rtphdr* rh,u_int *h261size)
{
	u_int h261header = htonl(*(u_int*)(rh+1));
	u_short seqno = ntohs(rh->rh_seqno);
	u_int rtflags = ntohs(rh->rh_flags);
	u_int j,i,x, l = seqno & H261_SLOTMASK;
	u_int	last,size,sbit;
	u_int firstlong,psc_mask,psc_word;
//	fprintf(stderr,"H261Decoder::"__FUNCTION__"(seqno=%d)\n",seqno);

	slot_[l].cc = cc;
	slot_[l].seqno = seqno;
	slot_[l].rtflags = rtflags;
	slot_[l].h261header = h261header;
	memcpy((char*)slot_[l].bp, (char*)bp, cc);

	/*
	 * find out, if we have a full stream in the queue
	 */
	for (i=0;i<H261_SLOTS;i++) {
		seqno = slot_[i].seqno;
		if (!FLAG_GOB(slot_[i].h261header)) { //this could be the start
			/* in the first bytes should be the 20 bits PSC */
			/* 0000 0000 0000 0001 0000 */
			u_char *p = slot_[i].bp;
			firstlong = (p[0]<<24)+(p[1]<<16)+(p[2]<<8)+p[3];
			psc_mask = 0xfffff000;
			psc_word = 0x00010000;
			for (j=0;j<8;j++) {
				if ((firstlong & psc_mask)==psc_word)
					break;
				psc_mask >>=1;
				psc_word >>=1;
			}
			if (j==8) continue; //there is no PSC. continue.

			for (j=0;j<H261_SLOTS;j++) {
				x=(i+j)&H261_SLOTMASK;
				if ((slot_[x].seqno&0xffff)!= (seqno+j)&0xffff)
					break;
				if (!slot_[x].cc)
					break;
				if (slot_[x].rtflags & RTP_M) {
					/* ENDE marker gefunden! */
					size = 0;
					for (l=0;l<=j;l++) {
						x=(i+l)&H261_SLOTMASK;
						size+=slot_[x].cc;
						if (FLAG_EBIT(slot_[x].h261header))
							size--;
					}
					size+=16;
					if (size>h261streamsize_) {
						delete[] h261stream_;
						h261stream_ = new u_char[size];
						h261streamsize_ = size;
					}
					h261stream_[0] = 0;
					last=0;
					sbit = FLAG_SBIT(slot_[i].h261header);
					/* need to set following two bits or we
					 * won't see anything on the output
					 */
					slot_[i].bp[(sbit+29)>>3]|=(1<<(7-((sbit+29)&7)));
					slot_[i].bp[(sbit+30)>>3]|=(1<<(7-((sbit+30)&7)));
					for (l=0;l<=j;l++) {
						x=(i+l)&H261_SLOTMASK;
						if (FLAG_SBIT(slot_[x].h261header)) {
							h261stream_[last]|=slot_[x].bp[0];
							last++;
							memcpy(h261stream_+last,slot_[x].bp+1,slot_[x].cc-1);
							last+=slot_[x].cc-1;
						} else {
							memcpy(h261stream_+last,slot_[x].bp,slot_[x].cc);
							last+=slot_[x].cc;
						}
						if ((l<j) && FLAG_EBIT(slot_[x].h261header))
							last--;
						slot_[x].cc = 0;
					}
/*
					for (l=0;l<=10;l++)
						fprintf(stderr,"%02x ",h261stream_[l]);
					fprintf(stderr,"\n");
 */
					*h261size = last;
					return 1;
				}
			}
		}
	}
	return 0;
}

void H261Decoder::recv(const rtphdr* rh, const u_char* bp, int cc)
{
	u_int	h261streamsize;
	u_int	v = ntohl(*(u_int*)(rh + 1));
	int	sbit = v >> 29;
	int	ebit = (v >> 26) & 7;
	int	quant = (v >> 10) & 0x1f;
	int	mvdh = (v >> 5) & 0x1f;
	int	mvdv = v & 0x1f;
	int	mba, gob;

	// clone a codec
	if (codec_ == 0) {
#if 0
	        u_char* vh = (u_char*)(rh + 1);
	        if ((vh[0] & 2) != 0) {
	                codec_ = new IntraP64Decoder();
	        } else {
	                codec_ = new FullP64Decoder();
		}
#endif
		// We always use the FullP64Decoder since above logic
		// fails sometimes. (dunno why) 

		codec_ = new FullP64Decoder();
	        codec_->marks(rvts_);
	}
	// simulate drop outs
/*
	if ((random() % 100) < 50) {
		fprintf(stderr,"-");
		return 0;
	} else 
		fprintf(stderr,"+");
 */
	/*
	 * vic-2.7 swapped the GOB and MBA fields in the RTP packet header
	 * with respect to the spec.  To maintain backward compat, whenever
	 * we see an out of range gob, we change our assumption about the
	 * stream and continue.
	 */
	if (!h261_rtp_bug_) {
		mba = (v >> 15) & 0x1f;
		gob = (v >> 20) & 0xf;
		if (gob > 12) {
			h261_rtp_bug_ = 1;
			mba = (v >> 19) & 0x1f;
			gob = (v >> 15) & 0xf;
		}
	} else {
		mba = (v >> 19) & 0x1f;
		gob = (v >> 15) & 0xf;
		if (gob > 12) {
			h261_rtp_bug_ = 0;
			mba = (v >> 15) & 0x1f;
                        gob = (v >> 20) & 0xf;
		}
	}

	if (gob > 12) {
		count(STAT_BAD_HEADER);
		return;
	}
	codec_->mark(now_);
	(void)codec_->decode(bp, cc, sbit, ebit, mba, gob, quant, mvdh, mvdv);
	/*
	 * If the stream changes format, issue a resize.
	 */
	if (codec_->width() != inw_) {
		resize(codec_->width(), codec_->height());
		codec_->marks(rvts_);
		count(STAT_FMT_CHANGE);
	}

	/*XXX*/
	if (ntohs(rh->rh_flags) & RTP_M) {
		codec_->sync();
		ndblk_ = codec_->ndblk();
		render_frame(codec_->frame());
		codec_->resetndblk();
	}

	if (!reassemble(bp, cc, rh, &h261streamsize ))
		return;
	H261Frame hf(0,(short*)h261stream_,h261streamsize,0,0);
	for (Renderer* p = engines_; p != 0; p = p->next_)
		if (p->ft() & FT_HW)
			p->consume(&hf);
}

void H261Decoder::redraw()
{
	if (codec_ != 0)
		Decoder::redraw(codec_->frame());
}
