/* MMSvideo  Copyright (c) 1994-1996 Rudolf Koenig &                        */
/*           Friedrich-Alexander Universitaet Erlangen-Nuernberg, IMMD4     */
/*           All rights reserved.                                           */

/*           THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF                 */
/*           Rudolf Koenig & FAU Erlangen-Nuernberg                         */
/*           The copyright notice above does not evidence any               */
/*           actual or intended publication of such source code.            */

#include <stdio.h>
#include "ras.h"
#include "image.h"

extern int errno;

#define RAS_ESC 128
#define FREEDATA  do { 				\
  if (!load_chunk)				\
    {						\
      free(im->data);				\
      if (im->cmap.red) free(im->cmap.red);	\
      if (im->cmap.green) free(im->cmap.green); \
      if (im->cmap.blue) free(im->cmap.blue);	\
    }						\
} while (0)

#define GETC(c,fp) do {			\
  if((c=getc(fp)) == EOF) 		\
    { 					\
      FREEDATA;				\
      if (!load_chunk)			\
	return IM_CORRUPTED;		\
      else				\
	return IM_OK;			\
    }					\
} while (0)

#define JGETC(c, fp, buf, len) do {	\
  if (fp) 				\
    GETC(c, fp);			\
  else 					\
    {					\
      if ((len)-- > 0)			\
        (c) = *(buf)++; 		\
      else				\
        return IM_CORRUPTED;		\
    }					\
} while (0)

/*
  valid bitmap_pads are: 8 16 24 32 ...
*/

/* call with either fp == NULL or { len == 0; buf == NULL} */
int
do_raster_load(fp, buf, len, im, bitmap_pad, load_chunk)
  FILE *fp;
  unsigned char *buf;
  int len;
  XP_Image *im;
  int load_chunk, bitmap_pad;
{
  struct rasterfile rf;
  int i, j, k, l, c = 0,
      bpl   = 0,   /* bytes per line + bitmap_pad in memory */
      r_bpl = 0, /* (real) bytes per line */
      f_bpl, /* bpl in the file */
      cn = 0, count;
  unsigned char *data, *dp;

  data = NULL;
  im->cmap.red = im->cmap.green = im->cmap.blue = 0;

  if (fp)
    {
      if(!fread((char *)&rf, sizeof(rf), 1, fp))
	return IM_CORRUPTED;
    }
  else
    {
      if ((len -= sizeof(rf)) < 0)
	return IM_CORRUPTED;
      bcopy(buf, (char *)&rf, sizeof(rf));
      buf += sizeof(rf);
    }

  rf.ras_magic     = LONGSWAP(rf.ras_magic);
  rf.ras_width     = LONGSWAP(rf.ras_width);
  rf.ras_height    = LONGSWAP(rf.ras_height);
  rf.ras_depth     = LONGSWAP(rf.ras_depth);
  rf.ras_length    = LONGSWAP(rf.ras_length);
  rf.ras_type      = LONGSWAP(rf.ras_type);
  rf.ras_maptype   = LONGSWAP(rf.ras_maptype);
  rf.ras_maplength = LONGSWAP(rf.ras_maplength);

  if(rf.ras_magic != RAS_MAGIC)
    return IM_UNKNOWN_TYPE;
  if(!(rf.ras_type == RT_STANDARD || rf.ras_type == RT_BYTE_ENCODED ||
       rf.ras_type == RT_YUV411) ||
     !(rf.ras_maptype == RMT_NONE || rf.ras_maptype == RMT_EQUAL_RGB) )
    return IM_UNSUPP_VERSION;
  im->depth = rf.ras_depth;
  if(!(im->depth == 1 || im->depth == 8 || im->depth == 24 || im->depth == 32))
    return IM_UNSUPP_DEPTH;
  if(! (rf.ras_width || rf.ras_height))
    return IM_CORRUPTED;

  switch(rf.ras_depth)
    {
      case 1:
	r_bpl = bpl = (rf.ras_width + 7) / 8;
	break;
      case 8:
        r_bpl = bpl = rf.ras_width;
	break;
      case 24:
      case 32:
	im->depth = 32;
	r_bpl = (rf.ras_depth / 8) * rf.ras_width;
	bpl = 4 * rf.ras_width;
	break;
    }
  f_bpl = (r_bpl + 1) & ~1;
  i = bpl % (bitmap_pad/8); 
  if (i)
    bpl += (bitmap_pad/8) - i;

  if (!(data = (unsigned char *)malloc(rf.ras_height * bpl)))
    return IM_NO_MEM;

  im->height = rf.ras_height;
  im->width = rf.ras_width;
  im->bytes_per_line = bpl;
  if(rf.ras_depth == 8)
    im->cmap.length = (256 * 3);
  else
    im->cmap.length = 0;
  im->cmap.grayscale = 0;
  im->mask.data = NULL;
  im->data = data;

  if(rf.ras_maptype == RMT_EQUAL_RGB)
    {
      if(rf.ras_maplength > 3 * 256)
	{
	  fprintf(stderr, "rasterload: maplength > 3 * 256, truncating...\n");
          rf.ras_maplength = 3 * 256;
	}
      im->cmap.red   = (unsigned char *)calloc(1, 256);
      im->cmap.green = (unsigned char *)calloc(1, 256);
      im->cmap.blue  = (unsigned char *)calloc(1, 256);
      if(!im->cmap.red || !im->cmap.green || !im->cmap.blue)
	{
	  FREEDATA;
	  return IM_NO_MEM; 
	}

      if (fp)
	{
	  if(!fread((char *)im->cmap.red, rf.ras_maplength / 3, 1, fp) ||
	     !fread((char *)im->cmap.green, rf.ras_maplength / 3, 1, fp) ||
	     !fread((char *)im->cmap.blue, rf.ras_maplength / 3, 1, fp))
	    {
	      FREEDATA;
	      return IM_CORRUPTED;
	    }
	}
      else
	{
	  if ((len -= rf.ras_maplength) < 0)
	    return IM_CORRUPTED;
	  bcopy(buf, (char *)im->cmap.red, rf.ras_maplength / 3);
	  buf += rf.ras_maplength / 3;
	  bcopy(buf, (char *)im->cmap.green, rf.ras_maplength / 3);
	  buf += rf.ras_maplength / 3;
	  bcopy(buf, (char *)im->cmap.blue, rf.ras_maplength / 3);
	  buf += rf.ras_maplength / 3;
	}

      if(im->cmap.length == 0)
	{
	  fprintf(stderr, "rasterload: image of depth %d with colormap\n",
					      im->depth);
	  free(im->cmap.red);
	  free(im->cmap.red);
	  free(im->cmap.red);
	  im->cmap.red = im->cmap.green = im->cmap.blue = 0;
	}
      else
	{
	  /* Check, if it's grayscale */
	  for(i = 0; i < 256; i++)
	    if(im->cmap.red[i] != im->cmap.green[i] ||
					 im->cmap.red[i] != im->cmap.blue[i])
	      break;
	  if(i == 256)
	    im->cmap.grayscale = 1;
	}
    }
  else if(im->depth == 8)
    {
      im->cmap.red   = (unsigned char *)calloc(1, 256);
      im->cmap.green = (unsigned char *)calloc(1, 256);
      im->cmap.blue  = (unsigned char *)calloc(1, 256);
      if(!im->cmap.red || !im->cmap.green || !im->cmap.blue)
	{
	  FREEDATA;
	  return IM_NO_MEM; 
	}
      for(i = 0; i < 256; i++)
	im->cmap.red[i] = im->cmap.green[i] = im->cmap.blue[i] = i;
      im->cmap.grayscale = 1;
    }

  if(rf.ras_type == RT_STANDARD)
    {
      for(i = 0; i < rf.ras_height; i++)
	{
          dp = data + i * bpl;
	  if(rf.ras_depth == 24)
	    {
	      for(j = 0; j < im->width; j++)
		{
		  *dp++ = 0;
		  JGETC(c, fp, buf, len); *dp++ = c;
		  JGETC(c, fp, buf, len); *dp++ = c;
		  JGETC(c, fp, buf, len); *dp++ = c;
		}
	      j = r_bpl;
	    }
	  else
	    {
	      for(j = 0; j < r_bpl; j++)
		{
		  JGETC(c, fp, buf, len);
		  *dp++ = c;
		}
	    }
	  for(;j < f_bpl; j++)
	    JGETC(c, fp, buf, len);
	}
    }
  else if(rf.ras_type == RT_BYTE_ENCODED)
    {
      dp = data;
      j = f_bpl;
      i = rf.ras_height;
      l = f_bpl % (bitmap_pad/8); 
      if (l)
	l = (bitmap_pad/8) - l;

      for (count = 0;;count--)
	{
	  if (count == 0)
	    {
	      JGETC(c, fp, buf, len);
	      count = 1;
	      if (c == RAS_ESC)
		{
	          JGETC(k, fp, buf, len);
		  if (k != 0)
		    {
		      count += k;
		      JGETC(c, fp, buf, len);
		    }
		}
	    }
	  if(rf.ras_depth == 24)
	    {
	      if(cn == 0)
		{
		  *dp++ = 0;
		  cn = 2;
		}
	      else
		cn--;
	    }
	  if (j > f_bpl - r_bpl)
	    *dp++ = c;
	  if (--j == 0)
	    {
	      j = f_bpl;
	      dp = (data += bpl);
	      if(--i == 0)
		break;
	      cn = 0;
	    }
	}
    }
  else /* RT_YUV411 */
    {
/* Let's read in...Y... */
      for(i = 0; i < rf.ras_height; i++)
	{
          dp = data + i * bpl;
	  for(j = 0; j < rf.ras_width; j++)
	    {
	      JGETC(c, fp, buf, len);
	      *dp = c; dp += 4;
	    }
	}
/* u, v */
      for(k = 1; k < 3; k++)
	for(i = 0; i < rf.ras_height; i += 2)
	  {
	    dp  = data + i * bpl + k;
	    for(j = 0; j < rf.ras_width; j += 2)
	      {
	        JGETC(c, fp, buf, len);
		*dp = c; dp += 8;
	      }
	  }
/* Convert yuv to rgb  (xbgr) */
      for(i = 0; i < rf.ras_height; i += 2)
	{
          dp  = data + i * bpl;
	  for(j = 0; j < rf.ras_width; j += 2)
	    {
	      double r_uv, g_uv, b_uv;
	      unsigned char y;

	      r_uv = 1.402 * *(dp+2);
	      g_uv = -(0.344 * *(dp+1) + 0.714 * *(dp+2));
	      b_uv = 1.772 * *(dp+1);

	      y = *dp;
	      *(dp+1) = y + b_uv; *(dp+2) = y + g_uv; *(dp+3) = y + r_uv;
	      y = *(dp+4);
	      *(dp+5) = y + b_uv; *(dp+6) = y + g_uv; *(dp+7) = y + r_uv;
	      dp += bpl;
	      y = *dp;
	      *(dp+1) = y + b_uv; *(dp+2) = y + g_uv; *(dp+3) = y + r_uv;
	      y = *(dp+4);
	      *(dp+5) = y + b_uv; *(dp+6) = y + g_uv; *(dp+7) = y + r_uv;

	      dp = dp - bpl + 8;
	    }
	}
    }
  return IM_OK;
}

int
raster_load(fp, im, bitmap_pad, load_chunk)
  FILE *fp;
  XP_Image *im;
  int load_chunk, bitmap_pad;
{
  return do_raster_load(fp, NULL, 0, im, bitmap_pad, load_chunk);
}

#define PUTC(a, b) { if(putc(a, b) == EOF) return IM_WRITE_FAILED; }
#define COUNT(x)   {if(what == x && count < 256) count++; else {\
		      if(write_be(fp, count, what))\
			return IM_WRITE_FAILED;\
		      count = 1; what = x;\
		   }}

/* Write a byte-encoded sequence */
static int
write_be(fp, count, what)
  FILE *fp;
  int count, what;
{
  int i;

  if(count < 3)
    {
      if(what != RAS_ESC)
        {
	  for(i = 0; i < count; i++)
	    PUTC(what, fp)
	}
      else
        {
	  PUTC(RAS_ESC, fp)
	  PUTC(count-1, fp)
	  if(count != 1)
	    PUTC(what, fp)
	}
    }
  else
    {
      PUTC(RAS_ESC, fp)
      PUTC(count-1, fp)
      PUTC(what, fp)
    }
  return IM_OK;
}

/*
 * if im->depth is 32, the empty bytes are written too
 */
int
raster_save(fp, im, type)
  FILE *fp;
  XP_Image *im;
  int type;
{
  struct rasterfile r;
  int i, j, fwidth, width = 0, count, what;
  unsigned char *d;

  switch(im->depth)
    {
      case 1:
        width = (im->width + 7) / 8;
	break;
      case 8:
        width = im->width;
        break;
      case 24:
      case 32:
        width = 4 * im->width;
        break;
      default:
	return IM_UNSUPP_DEPTH;
    }
  if(im->depth == 24)
    fwidth = 3 * im->width;
  else
    fwidth = width;
  fwidth = (fwidth+1) & ~1;

  r.ras_magic = RAS_MAGIC;
  r.ras_width = im->width;
  r.ras_height = im->height;
  r.ras_depth = im->depth;
  r.ras_length = fwidth * im->height;
  r.ras_type = type;
  /* we may have unused color-cells here: no optimisation. snif. */
  r.ras_maplength = im->cmap.length;
  r.ras_maptype = im->cmap.length ? RMT_EQUAL_RGB : RMT_NONE;

  r.ras_magic     = LONGSWAP(r.ras_magic);
  r.ras_width     = LONGSWAP(r.ras_width);
  r.ras_height    = LONGSWAP(r.ras_height);
  r.ras_depth     = LONGSWAP(r.ras_depth);
  r.ras_length    = LONGSWAP(r.ras_length);
  r.ras_type      = LONGSWAP(r.ras_type);
  r.ras_maptype   = LONGSWAP(r.ras_maptype);
  r.ras_maplength = LONGSWAP(r.ras_maplength);

  if(fwrite((char *)&r, sizeof(r), 1, fp) != 1)
    return IM_WRITE_FAILED;

  r.ras_magic     = LONGSWAP(r.ras_magic);
  r.ras_width     = LONGSWAP(r.ras_width);
  r.ras_height    = LONGSWAP(r.ras_height);
  r.ras_depth     = LONGSWAP(r.ras_depth);
  r.ras_length    = LONGSWAP(r.ras_length);
  r.ras_type      = LONGSWAP(r.ras_type);
  r.ras_maptype   = LONGSWAP(r.ras_maptype);
  r.ras_maplength = LONGSWAP(r.ras_maplength);

  if(r.ras_maplength)
    {
      if(fwrite(im->cmap.red, r.ras_maplength/3, 1, fp) != 1)
        return IM_WRITE_FAILED;
      if(fwrite(im->cmap.green, r.ras_maplength/3, 1, fp) != 1)
        return IM_WRITE_FAILED;
      if(fwrite(im->cmap.blue, r.ras_maplength/3, 1, fp) != 1)
        return IM_WRITE_FAILED;
    }

  if(type == RT_BYTE_ENCODED)
    {
      count = 0; what = -1;
      for(i = 0; i < im->height; i++)
	{
          d = (unsigned char *)(im->data + i * im->bytes_per_line);
	  for(j = 0; j < width; j++, d++)
	    {
	      if(im->depth == 24 && (j % 4) == 0)
	        continue;
	      COUNT(*d)
	    }
	  if(width & 1)
	    COUNT(0)
	}
      if(write_be(fp, count, what))
	return IM_WRITE_FAILED;
    }
  else /* RT_STANDARD */
    {
      for(i = 0; i < im->height; i++)
	{
          d = im->data + i * im->bytes_per_line;
	  for(j = 0; j < width; j++, d++)
	    {
	      if(im->depth == 24 && (j % 4) == 0)
	        continue;
	      PUTC(*d, fp)
	    }
/* Write a padding character */
	  if(width & 1)
	    PUTC(0, fp)
	}
    }
  return IM_OK;
}
