/*-
 * Reassembly part
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
#include "module.h"
#include "decoder.h"
#include "renderer.h"
#include "bsd-endian.h"
#include "Tcl.h"
#include "jpeg/jpeg.h"
#include "reass.h"

/*
 * Initial size of each reassembly buffer.
 */
#define JPEG_BUFSIZE (16*1024)

#define STAT_BADOFF 0
#define STAT_HUGEFRM 1

Sr_JpegReassembler::Sr_JpegReassembler()
	: decimate_(0), ndec_(0), hugefrm_(0), badoff_(0)
{
	rbsize_ = JPEG_BUFSIZE;
	rb0_.bp = new u_char[2 * JPEG_BUFSIZE];
	rb0_.ts = ~0;
	rb0_.drop = 0;
	rb1_.bp = &rb0_.bp[JPEG_BUFSIZE];
	rb1_.ts = ~0;
	rb1_.drop = 0;
	memset(slots_, 0, sizeof(slots_));
}

Sr_JpegReassembler::~Sr_JpegReassembler()
{
	delete rb0_.bp;
}

/*
 * Reassemble an RTP/JPEG stream.  Return a pointer to a buffer
 * each time we encounter an entire frame.  Otherwise, return 0.
 * Set len to the length of the jpeg data in the buffer.
 */
u_char* Sr_JpegReassembler::reassemble(const rtphdr* rh, const u_char* bp, int& len)
{
	jpeghdr* p = (jpeghdr*)(rh + 1);
	//mms_ctl_parallax_jpeg* p = (mms_ctl_parallax_jpeg*)(rh + 1);
	int off = (int)ntohl(p->off);
	int cc = len;
	if (off < 0) {
		++badoff_;
		return (0);
	}
	if (off + cc > rbsize_) {
		/*
		 * Check for outrageous frame size.
		 */
		if (off + cc > 250*1024) {
			++hugefrm_;
			return (0);
		}
		/*
		 * Grow reassembly buffers.
		 */
		int nsize = rbsize_;
		do {
			nsize <<= 1;
		} while (off + cc > nsize);
		u_char* p = new u_char[2 * nsize];
		memcpy(p, rb0_.bp, rbsize_);
		memcpy(p + nsize, rb1_.bp, rbsize_);
		delete rb0_.bp;
		rb0_.bp = p;
		rb1_.bp = p + nsize;
		rbsize_ = nsize;
	}
	/*
	 * Initialize the slot data structure.
	 */
	int seqno = ntohs(rh->rh_seqno);
	int s = seqno & JPEG_SLOTMASK;
	u_int32_t ts = ntohl(rh->rh_ts);
	slots_[s].seqno = seqno;
	slots_[s].off = off;
	slots_[s].ts = ts;
	/*
	 * Figure out which reassembly-buffer to use.  If we're not
	 * already reassembling this frame, take over the older buffer.
	 */
	rbuf* rb;
	if (ts == rb0_.ts)
		rb = &rb0_;
	else if (ts == rb1_.ts)
		rb = &rb1_;
	else {
		rb = ((int)(rb0_.ts - rb1_.ts) < 0) ? &rb0_ : &rb1_;
		rb->ts = ts;
		rb->drop = 0;
		/*
		 * If we're decimating frames (to save cycles),
		 * remember that we might want to drop the rest
		 * of the packets from this frame.
		 */
		if (decimate_) {
			if (--ndec_ <= 0)
				ndec_ = decimate_;
			else
				rb->drop = 1;
		}
	}
	if (rb->drop)
		return (0);

	memcpy((char*)&rb->bp[off], (char*)bp, cc);

	/*
	 * Check if we're at end-of-frame.  If not, see if we're
	 * filling a hole.  If not, return.  Otherwise, drop out
	 * below and check for an entire frame.  We set cc to be
	 * the entire frame size in the if-else below.
	 */
	if ((ntohs(rh->rh_flags) & RTP_M) != 0) {
		slots_[s].eof = cc;
		cc += off;
	} else {
		slots_[s].eof = 0;
		int ns = s;
		do {
			ns = (ns + 1) & JPEG_SLOTMASK;
			if (slots_[ns].ts != ts || ns == s)
				return (0);
		} while (slots_[ns].eof != 0);
		cc = int(slots_[ns].eof + slots_[ns].off);
	}
	/*
	 * At this point, we know we have an end-of-frame, and
	 * all packets from slot 's' up until the end-of-frame.
	 * Scan backward from slot 's' making sure we have all
	 * packets from the start-of-frame (off == 0) to 's'.
	 */
	int ps = s;
	do {
		ps = (ps - 1) & JPEG_SLOTMASK;
		if (slots_[ps].ts != ts || ps == s)
			return (0);
	} while (slots_[ps].off != 0);

	len = cc;
	return (rb->bp);
}
