#include "os.h"
#include "tcb.h"
#include "os/scheduler/task.h"
#include "os/scheduler/tasklist.h"
#include "os/scheduler/scheduler.h"
#include "os/alarm.h"

extern "C" void OSEKOS_TASK_H1();
extern "C" void OSEKOS_TASK_H2();
extern "C" void OSEKOS_TASK_H3();
extern "C" void StartOS(int);
extern "C" StatusType OSEKOS_TerminateTask__ABB456();
extern "C" StatusType OSEKOS_TerminateTask__ABB465();
extern "C" StatusType OSEKOS_ActivateTask__ABB460(TaskType);
extern "C" StatusType OSEKOS_TerminateTask__ABB463();
void __OS_HOOK_PreIdleHook();

extern "C" uint8_t H1_stack[4096];
extern "C" uint8_t H2_stack[4096];
extern "C" uint8_t H3_stack[4096];
namespace os{
    extern Counter OS_C1_counter;
    extern Alarm OS_A1_alarm;
}

namespace os{
    namespace tasks{
        extern const os::scheduler::Task OS_H1_task;
        extern const os::scheduler::Task OS_H2_task;
        extern const os::scheduler::Task OS_H3_task;
    }
}

namespace arch{
    extern void * OS_H1_stackptr;
    extern void * OS_H2_stackptr;
    extern void * OS_H3_stackptr;
    extern const arch::TCB OS_H1_tcb;
    extern const arch::TCB OS_H2_tcb;
    extern const arch::TCB OS_H3_tcb;
}


uint8_t H1_stack[4096];
uint8_t H2_stack[4096];
uint8_t H3_stack[4096];
namespace os{
    Counter OS_C1_counter(1000, 1, 1);
    Alarm OS_A1_alarm(OS_C1_counter, os::tasks::OS_H2_task, true, 100, 10);
}

namespace os{
    namespace tasks{
        constexpr const os::scheduler::Task OS_H1_task(1, 3, true, arch::OS_H1_tcb);
        constexpr const os::scheduler::Task OS_H2_task(2, 2, true, arch::OS_H2_tcb);
        constexpr const os::scheduler::Task OS_H3_task(3, 1, true, arch::OS_H3_tcb);
    }
}

namespace arch{
    void * OS_H1_stackptr;
    void * OS_H2_stackptr;
    void * OS_H3_stackptr;
    constexpr const arch::TCB OS_H1_tcb(&OSEKOS_TASK_H1, H1_stack, OS_H1_stackptr, 4096);
    constexpr const arch::TCB OS_H2_tcb(&OSEKOS_TASK_H2, H2_stack, OS_H2_stackptr, 4096);
    constexpr const arch::TCB OS_H3_tcb(&OSEKOS_TASK_H3, H3_stack, OS_H3_stackptr, 4096);
}

using namespace os;
using namespace os::tasks;
using namespace arch;

// ****************************************************************
// TaskList Implementation
// ****************************************************************

namespace os { namespace scheduler {

// only needed for Tasklist constructor comma placement
class UnencodedTaskListStatic {};

/* Simpler array based task queue */
struct TaskList : public UnencodedTaskListStatic {
    typedef uint8_t id_t;
    typedef uint8_t prio_t;

    // encoded task priorities
    
    prio_t H1;
    prio_t H2;
    prio_t H3;
    

    // idle task id/priority
    static constexpr id_t idle_id = 0;
    static constexpr prio_t idle_prio = 0;

    
    TaskList() : UnencodedTaskListStatic() , H1(0), H2(0), H3(0) {}

    /** Set priority of task id to prio **/
    forceinline void set(const prio_t id, const prio_t prio) {

         if(id == 1) {
             H1 = prio;
         } else if(id == 2) {
             H2 = prio;
         } else if(id == 3) {
             H3 = prio;
         } else {
             assert(false);
         }
    }

    /** Set priority of task id to prio **/
    forceinline void increasePrio(const prio_t id, const prio_t prio) {
        
        if(id == 1) {
            if (H1 < prio)
                H1 = prio;
        } else if(id == 2) {
            if (H2 < prio)
                H2 = prio;
        } else if(id == 3) {
            if (H3 < prio)
                H3 = prio;
        } else {
            assert(false);
        }
    }

    forceinline void head(id_t& id, prio_t& prio) const {
        // start with idle id/priority
        id = idle_id;
        prio = idle_prio;



        
        
                if(H1 > prio) {
                    prio = H1;
                    id = 1;
                }
        
        
                if(H2 > prio) {
                    prio = H2;
                    id = 2;
                }
        
        
                if(H3 > prio) {
                    prio = H3;
                    id = 3;
                }
        
        
    }

    forceinline void insert(const id_t& id, const prio_t& prio) {
        increasePrio(id, prio);
    }

    forceinline void remove(const id_t& id) {
        set(id, 0);
    }

    forceinline void promote(const id_t& id, const prio_t& newprio) {
        set(id, newprio);
    }
};

}; // scheduler
}; // os
namespace os {
namespace scheduler {

using namespace os::tasks;

struct Scheduler {
    os::scheduler::TaskList tlist;

    typedef uint8_t prio_t;
    typedef uint8_t id_t;

    prio_t current_prio;
    id_t current_task;

    static constexpr prio_t scheduler_prio = 4;

    forceinline void Reschedule(void) {
        // OPTIMIZATION: do not reschedule if RES_SCHEDULER is taken
        if (current_prio != scheduler_prio) {
            // set current (=next) task from task list
            tlist.head(current_task, current_prio);
        }

        // dispatch or enter idle
        // TODO: generated signature

        if(current_task == OS_H1_task.id) {
            if (OS_H1_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_H1_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_H1_task);
        } else if(current_task == OS_H2_task.id) {
            if (OS_H2_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_H2_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_H2_task);
        } else if(current_task == OS_H3_task.id) {
            if (OS_H3_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_H3_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_H3_task);
        } else if(current_task == TaskList::idle_id) {
            Dispatcher::idle();
        } else {
            assert(false);
        }
    }


    forceinline void SetReady_impl(const Task task) {
        IncreasePriority(task, task.prio);
    }


    forceinline void Schedule_impl(void) {
        if(in_syscall()) {
            // in syscall: reschedule directly
            Reschedule();
        } else {
            // not in syscall (probably in ISR): request reschedule AST
            // Calls also Reschedule()
            request_reschedule_ast();
        }
    }

    forceinline void SetSuspended_impl(const Task t) {
        t.tcb.reset();

        tlist.remove(t.id);
        if (t.preemptable == false) {
            // restore non-preemptable task to original priority
            // this is required for the optimization in Reschedule()
            current_prio = t.prio;
        }
    }

    forceinline void ActivateTask_impl(const Task t) {
        SetReady_impl(t);
        Schedule_impl();
    }

    forceinline void ChainTask_impl(const Task from, const Task to) {
        auto from_id = from.id;
        assert(from_id == current_task);

        SetSuspended_impl(from);
        SetReady_impl(to);
        Schedule_impl();
    }

    forceinline void TerminateTask_impl(const Task from) {
        auto id = from.id;
        assert(id == current_task);

        SetSuspended_impl(from);

        Schedule_impl();
    }

    forceinline void GetResource_impl(const Task current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
    }

    forceinline void ReleaseResource_impl(const Task current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
        Schedule_impl();
    }


    forceinline void SetReadyFromSuspended_impl(const Task task) {
        SetPriority(task, task.prio);
    }

    // Low level interface to the task list
    forceinline void SetCurrentTask(const Task task) {
        current_task = task.id;
    }

    forceinline void SetSystemPriority(const int new_prio) {
        current_prio = new_prio;
    }

    forceinline void SetPriority(const Task task, const int new_prio) {
        tlist.set(task.id, new_prio);
    }

    forceinline void IncreasePriority(const Task task, const int new_prio) {
        

        if (task.id == OS_H1_task.id) {
            tlist.insert(OS_H1_task.id,  new_prio);
        } else if (task.id == OS_H2_task.id) {
            tlist.insert(OS_H2_task.id,  new_prio);
        } else if (task.id == OS_H3_task.id) {
            tlist.insert(OS_H3_task.id,  new_prio);
        } else {
            assert(false);
        }
    }

};

constexpr Scheduler::prio_t Scheduler::scheduler_prio;
Scheduler scheduler_;

noinline void ScheduleC_impl(__attribute__ ((unused)) uint32_t dummy) {
    scheduler_.Reschedule();
}

}; // scheduler
}; // os
#include "counter.h"
#include "alarm.h"

namespace os {

void inlinehint Counter::tick() {
    
    if(OS_C1_counter.value == OS_C1_counter.maxallowedvalue) {
        OS_C1_counter.value = 0;
    } else {
        OS_C1_counter.value++;
    }
    Alarm::checkCounter(OS_C1_counter);
    
}

void inlinehint Alarm::checkCounter(const Counter &counter) {
    (void) counter; // unused argument warning might happen

    
    
    
    
    if ((&counter == &OS_C1_counter) && OS_A1_alarm.checkTrigger(&OS_C1_counter)) {
        debug << "Alarm triggered" << endl;
        os::scheduler::scheduler_.ActivateTask_impl(*OS_A1_alarm.task_);
    } 
    
}


} // os

extern "C" void StartOS(int arg0){
    (void) arg0;
    OS_H1_task.tcb.reset();
    OS_H2_task.tcb.reset();
    OS_H3_task.tcb.reset();
    Machine::enable_interrupts();
    syscall(os::scheduler::ScheduleC_impl, 0);
    Machine::unreachable();
}

extern "C" StatusType OSEKOS_TerminateTask__ABB456(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.TerminateTask_impl(OS_H1_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_TerminateTask__ABB465(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_H3_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask Idle>
         */
        scheduler_.current_task = TaskList::idle_id;
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 0 == <Subtask Idle>
         */
        scheduler_.SetSystemPriority(0);
        Dispatcher::idle();
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_ActivateTask__ABB460(TaskType arg0){
    // Hook: SystemEnterHook
    {
    }
    (void) arg0;
    Machine::disable_interrupts();
    {
        scheduler_.SetReady_impl(OS_H3_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_H2>
         */
        // OPTIMIZATION: We do not have to update the current_task entry
        // OPTIMIZATION: We do not have to update the current system priority
        // OPTIMIZATION: The task is surely ready, just resume it.
        Dispatcher::ResumeToTask(OS_H2_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_TerminateTask__ABB463(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_H2_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_H3>
         */
        scheduler_.SetCurrentTask(OS_H3_task);
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 1 == <Subtask OSEKOS_TASK_H3>
         */
        scheduler_.SetSystemPriority(1);
        Dispatcher::Dispatch(OS_H3_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

void __OS_HOOK_PreIdleHook(){
    __OS_HOOK_DEFINED_PreIdleHook();
}

