#include "os.h"
#include "tcb.h"
#include "os/scheduler/task.h"
#include "os/scheduler/tasklist.h"
#include "os/scheduler/scheduler.h"
#include "os/alarm.h"

extern "C" void OSEKOS_TASK_Task1();
extern "C" void OSEKOS_TASK_Task2();
extern "C" void OSEKOS_TASK_Task3();
extern "C" void OSEKOS_TASK_Task4();
extern "C" void StartOS(int);
extern "C" StatusType OSEKOS_TerminateTask__ABB941();
extern "C" StatusType OSEKOS_TerminateTask__ABB598();
extern "C" StatusType OSEKOS_ActivateTask__ABB454(TaskType);
extern "C" StatusType OSEKOS_ActivateTask__ABB529(TaskType);
extern "C" StatusType OSEKOS_TerminateTask__ABB843();
extern "C" StatusType OSEKOS_ChainTask__ABB763(TaskType);
void __OS_HOOK_PreIdleHook();

extern "C" uint8_t Task1_stack[4096];
extern "C" uint8_t Task2_stack[4096];
extern "C" uint8_t Task3_stack[4096];
extern "C" uint8_t Task4_stack[4096];
namespace os{
    namespace tasks{
        extern const os::scheduler::Task OS_Task1_task;
        extern const os::scheduler::Task OS_Task2_task;
        extern const os::scheduler::Task OS_Task3_task;
        extern const os::scheduler::Task OS_Task4_task;
    }
}

namespace arch{
    extern void * OS_Task1_stackptr;
    extern void * OS_Task2_stackptr;
    extern void * OS_Task3_stackptr;
    extern void * OS_Task4_stackptr;
    extern const arch::TCB OS_Task1_tcb;
    extern const arch::TCB OS_Task2_tcb;
    extern const arch::TCB OS_Task3_tcb;
    extern const arch::TCB OS_Task4_tcb;
}


uint8_t Task1_stack[4096];
uint8_t Task2_stack[4096];
uint8_t Task3_stack[4096];
uint8_t Task4_stack[4096];
namespace os{
    namespace tasks{
        constexpr const os::scheduler::Task OS_Task1_task(1, 2, true, arch::OS_Task1_tcb);
        constexpr const os::scheduler::Task OS_Task2_task(2, 3, true, arch::OS_Task2_tcb);
        constexpr const os::scheduler::Task OS_Task3_task(3, 1, true, arch::OS_Task3_tcb);
        constexpr const os::scheduler::Task OS_Task4_task(4, 4, true, arch::OS_Task4_tcb);
    }
}

namespace arch{
    void * OS_Task1_stackptr;
    void * OS_Task2_stackptr;
    void * OS_Task3_stackptr;
    void * OS_Task4_stackptr;
    constexpr const arch::TCB OS_Task1_tcb(&OSEKOS_TASK_Task1, Task1_stack, OS_Task1_stackptr, 4096);
    constexpr const arch::TCB OS_Task2_tcb(&OSEKOS_TASK_Task2, Task2_stack, OS_Task2_stackptr, 4096);
    constexpr const arch::TCB OS_Task3_tcb(&OSEKOS_TASK_Task3, Task3_stack, OS_Task3_stackptr, 4096);
    constexpr const arch::TCB OS_Task4_tcb(&OSEKOS_TASK_Task4, Task4_stack, OS_Task4_stackptr, 4096);
}

using namespace os::tasks;
using namespace arch;

// ****************************************************************
// TaskList Implementation
// ****************************************************************

namespace os { namespace scheduler {

// only needed for Tasklist constructor comma placement
class UnencodedTaskListStatic {};

/* Simpler array based task queue */
struct TaskList : public UnencodedTaskListStatic {
    typedef uint8_t id_t;
    typedef uint8_t prio_t;

    // encoded task priorities
    
    prio_t Task1;
    prio_t Task2;
    prio_t Task3;
    prio_t Task4;
    

    // idle task id/priority
    static constexpr id_t idle_id = 0;
    static constexpr prio_t idle_prio = 0;

    
    TaskList() : UnencodedTaskListStatic() , Task1(0), Task2(0), Task3(0), Task4(0) {}

    /** Set priority of task id to prio **/
    forceinline void set(const prio_t id, const prio_t prio) {

         if(id == 1) {
             Task1 = prio;
         } else if(id == 2) {
             Task2 = prio;
         } else if(id == 3) {
             Task3 = prio;
         } else if(id == 4) {
             Task4 = prio;
         } else {
             assert(false);
         }
    }

    /** Set priority of task id to prio **/
    forceinline void increasePrio(const prio_t id, const prio_t prio) {
        
        if(id == 1) {
            if (Task1 < prio)
                Task1 = prio;
        } else if(id == 2) {
            if (Task2 < prio)
                Task2 = prio;
        } else if(id == 3) {
            if (Task3 < prio)
                Task3 = prio;
        } else if(id == 4) {
            if (Task4 < prio)
                Task4 = prio;
        } else {
            assert(false);
        }
    }

    forceinline void head(id_t& id, prio_t& prio) const {
        // start with idle id/priority
        id = idle_id;
        prio = idle_prio;



        
        
                if(Task1 > prio) {
                    prio = Task1;
                    id = 1;
                }
        
        
                if(Task2 > prio) {
                    prio = Task2;
                    id = 2;
                }
        
        
                if(Task3 > prio) {
                    prio = Task3;
                    id = 3;
                }
        
        
                if(Task4 > prio) {
                    prio = Task4;
                    id = 4;
                }
        
        
    }

    forceinline void insert(const id_t& id, const prio_t& prio) {
        increasePrio(id, prio);
    }

    forceinline void remove(const id_t& id) {
        set(id, 0);
    }

    forceinline void promote(const id_t& id, const prio_t& newprio) {
        set(id, newprio);
    }
};

}; // scheduler
}; // os
namespace os {
namespace scheduler {

using namespace os::tasks;

struct Scheduler {
    os::scheduler::TaskList tlist;

    typedef uint8_t prio_t;
    typedef uint8_t id_t;

    prio_t current_prio;
    id_t current_task;

    static constexpr prio_t scheduler_prio = 5;

    forceinline void Reschedule(void) {
        // OPTIMIZATION: do not reschedule if RES_SCHEDULER is taken
        if (current_prio != scheduler_prio) {
            // set current (=next) task from task list
            tlist.head(current_task, current_prio);
        }

        // dispatch or enter idle
        // TODO: generated signature

        if(current_task == OS_Task1_task.id) {
            if (OS_Task1_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Task1_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Task1_task);
        } else if(current_task == OS_Task2_task.id) {
            if (OS_Task2_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Task2_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Task2_task);
        } else if(current_task == OS_Task3_task.id) {
            if (OS_Task3_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Task3_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Task3_task);
        } else if(current_task == OS_Task4_task.id) {
            if (OS_Task4_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Task4_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Task4_task);
        } else if(current_task == TaskList::idle_id) {
            Dispatcher::idle();
        } else {
            assert(false);
        }
    }


    forceinline void SetReady_impl(const Task task) {
        IncreasePriority(task, task.prio);
    }


    forceinline void Schedule_impl(void) {
        if(in_syscall()) {
            // in syscall: reschedule directly
            Reschedule();
        } else {
            // not in syscall (probably in ISR): request reschedule AST
            // Calls also Reschedule()
            request_reschedule_ast();
        }
    }

    forceinline void SetSuspended_impl(const Task t) {
        t.tcb.reset();

        tlist.remove(t.id);
        if (t.preemptable == false) {
            // restore non-preemptable task to original priority
            // this is required for the optimization in Reschedule()
            current_prio = t.prio;
        }
    }

    forceinline void ActivateTask_impl(const Task t) {
        SetReady_impl(t);
        Schedule_impl();
    }

    forceinline void ChainTask_impl(const Task from, const Task to) {
        auto from_id = from.id;
        assert(from_id == current_task);

        SetSuspended_impl(from);
        SetReady_impl(to);
        Schedule_impl();
    }

    forceinline void TerminateTask_impl(const Task from) {
        auto id = from.id;
        assert(id == current_task);

        SetSuspended_impl(from);

        Schedule_impl();
    }

    forceinline void GetResource_impl(const Task current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
    }

    forceinline void ReleaseResource_impl(const Task current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
        Schedule_impl();
    }


    forceinline void SetReadyFromSuspended_impl(const Task task) {
        SetPriority(task, task.prio);
    }

    // Low level interface to the task list
    forceinline void SetCurrentTask(const Task task) {
        current_task = task.id;
    }

    forceinline void SetSystemPriority(const int new_prio) {
        current_prio = new_prio;
    }

    forceinline void SetPriority(const Task task, const int new_prio) {
        tlist.set(task.id, new_prio);
    }

    forceinline void IncreasePriority(const Task task, const int new_prio) {
        

        if (task.id == OS_Task1_task.id) {
            tlist.insert(OS_Task1_task.id,  new_prio);
        } else if (task.id == OS_Task2_task.id) {
            tlist.insert(OS_Task2_task.id,  new_prio);
        } else if (task.id == OS_Task3_task.id) {
            tlist.insert(OS_Task3_task.id,  new_prio);
        } else if (task.id == OS_Task4_task.id) {
            tlist.insert(OS_Task4_task.id,  new_prio);
        } else {
            assert(false);
        }
    }

};

constexpr Scheduler::prio_t Scheduler::scheduler_prio;
Scheduler scheduler_;

noinline void ScheduleC_impl(__attribute__ ((unused)) uint32_t dummy) {
    scheduler_.Reschedule();
}

}; // scheduler
}; // os
#include "counter.h"
#include "alarm.h"

namespace os {

void inlinehint Counter::tick() {
    
    
}

void inlinehint Alarm::checkCounter(const Counter &counter) {
    (void) counter; // unused argument warning might happen

    
    
    
    
    
}


} // os

extern "C" void StartOS(int arg0){
    (void) arg0;
    OS_Task1_task.tcb.reset();
    scheduler_.SetReadyFromSuspended_impl(OS_Task1_task);
    OS_Task2_task.tcb.reset();
    OS_Task3_task.tcb.reset();
    OS_Task4_task.tcb.reset();
    Machine::enable_interrupts();
    syscall(os::scheduler::ScheduleC_impl, 0);
    Machine::unreachable();
}

extern "C" StatusType OSEKOS_TerminateTask__ABB941(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_Task4_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Task1>
         */
        scheduler_.SetCurrentTask(OS_Task1_task);
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 2 == <Subtask OSEKOS_TASK_Task1>
         */
        scheduler_.SetSystemPriority(2);
        // OPTIMIZATION: The task is surely ready, just resume it.
        Dispatcher::ResumeToTask(OS_Task1_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_TerminateTask__ABB598(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_Task1_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Task3>
         */
        scheduler_.SetCurrentTask(OS_Task3_task);
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 1 == <Subtask OSEKOS_TASK_Task3>
         */
        scheduler_.SetSystemPriority(1);
        // OPTIMIZATION: The task is surely no paused. We can surely start it from scratch.
        Dispatcher::StartToTask(OS_Task3_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_ActivateTask__ABB454(TaskType arg0){
    // Hook: SystemEnterHook
    {
    }
    (void) arg0;
    Machine::disable_interrupts();
    {
        // OPTIMIZATION: We surely know that the task is suspended
        scheduler_.SetReadyFromSuspended_impl(OS_Task2_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Task2>
         */
        scheduler_.SetCurrentTask(OS_Task2_task);
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 3 == <Subtask OSEKOS_TASK_Task2>
         */
        scheduler_.SetSystemPriority(3);
        // OPTIMIZATION: The task is surely no paused. We can surely start it from scratch.
        Dispatcher::StartToTask(OS_Task2_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_ActivateTask__ABB529(TaskType arg0){
    // Hook: SystemEnterHook
    {
    }
    (void) arg0;
    Machine::disable_interrupts();
    {
        // OPTIMIZATION: We surely know that the task is suspended
        scheduler_.SetReadyFromSuspended_impl(OS_Task3_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Task1>
         */
        // OPTIMIZATION: We do not have to update the current_task entry
        // OPTIMIZATION: We do not have to update the current system priority
        // OPTIMIZATION: The task is surely ready, just resume it.
        Dispatcher::ResumeToTask(OS_Task1_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_TerminateTask__ABB843(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_Task3_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask Idle>
         */
        scheduler_.current_task = TaskList::idle_id;
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 0 == <Subtask Idle>
         */
        scheduler_.SetSystemPriority(0);
        Dispatcher::idle();
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_ChainTask__ABB763(TaskType arg0){
    // Hook: SystemEnterHook
    {
    }
    (void) arg0;
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_Task2_task);
        // OPTIMIZATION: We surely know that the task is suspended
        scheduler_.SetReadyFromSuspended_impl(OS_Task4_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Task4>
         */
        scheduler_.SetCurrentTask(OS_Task4_task);
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 4 == <Subtask OSEKOS_TASK_Task4>
         */
        scheduler_.SetSystemPriority(4);
        // OPTIMIZATION: The task is surely no paused. We can surely start it from scratch.
        Dispatcher::StartToTask(OS_Task4_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

void __OS_HOOK_PreIdleHook(){
}

