#include "os.h"
#include "tcb.h"
#include "os/scheduler/task.h"
#include "os/scheduler/tasklist.h"
#include "os/scheduler/scheduler.h"
#include "os/alarm.h"

extern "C" void OSEKOS_TASK_Handler12();
extern "C" void OSEKOS_TASK_Handler13();
extern "C" void OSEKOS_TASK_Handler11();
extern "C" void StartOS(int);
extern "C" StatusType OSEKOS_TerminateTask__ABB490();
extern "C" StatusType OSEKOS_TerminateTask__ABB486();
extern "C" StatusType OSEKOS_ActivateTask__ABB456(TaskType);
extern "C" StatusType OSEKOS_ActivateTask__ABB466(TaskType);
extern "C" StatusType OSEKOS_ActivateTask__ABB472(TaskType);
extern "C" StatusType OSEKOS_TerminateTask__ABB479();
void __OS_HOOK_PreIdleHook();

extern "C" uint8_t Handler12_stack[4096];
extern "C" uint8_t Handler13_stack[4096];
extern "C" uint8_t Handler11_stack[4096];
namespace os{
    namespace tasks{
        extern const os::scheduler::Task OS_Handler12_task;
        extern const os::scheduler::Task OS_Handler13_task;
        extern const os::scheduler::Task OS_Handler11_task;
    }
}

namespace arch{
    extern void * OS_Handler12_stackptr;
    extern void * OS_Handler13_stackptr;
    extern void * OS_Handler11_stackptr;
    extern const arch::TCB OS_Handler12_tcb;
    extern const arch::TCB OS_Handler13_tcb;
    extern const arch::TCB OS_Handler11_tcb;
}


uint8_t Handler12_stack[4096];
uint8_t Handler13_stack[4096];
uint8_t Handler11_stack[4096];
namespace os{
    namespace tasks{
        constexpr const os::scheduler::Task OS_Handler12_task(1, 1, true, arch::OS_Handler12_tcb);
        constexpr const os::scheduler::Task OS_Handler13_task(2, 3, true, arch::OS_Handler13_tcb);
        constexpr const os::scheduler::Task OS_Handler11_task(3, 2, true, arch::OS_Handler11_tcb);
    }
}

namespace arch{
    void * OS_Handler12_stackptr;
    void * OS_Handler13_stackptr;
    void * OS_Handler11_stackptr;
    constexpr const arch::TCB OS_Handler12_tcb(&OSEKOS_TASK_Handler12, Handler12_stack, OS_Handler12_stackptr, 4096);
    constexpr const arch::TCB OS_Handler13_tcb(&OSEKOS_TASK_Handler13, Handler13_stack, OS_Handler13_stackptr, 4096);
    constexpr const arch::TCB OS_Handler11_tcb(&OSEKOS_TASK_Handler11, Handler11_stack, OS_Handler11_stackptr, 4096);
}

using namespace os::tasks;
using namespace arch;

// ****************************************************************
// TaskList Implementation
// ****************************************************************

namespace os { namespace scheduler {

// only needed for Tasklist constructor comma placement
class UnencodedTaskListStatic {};

/* Simpler array based task queue */
struct TaskList : public UnencodedTaskListStatic {
    typedef uint8_t id_t;
    typedef uint8_t prio_t;

    // encoded task priorities
    
    prio_t Handler12;
    prio_t Handler13;
    prio_t Handler11;
    

    // idle task id/priority
    static constexpr id_t idle_id = 0;
    static constexpr prio_t idle_prio = 0;

    
    TaskList() : UnencodedTaskListStatic() , Handler12(0), Handler13(0), Handler11(0) {}

    /** Set priority of task id to prio **/
    forceinline void set(const prio_t id, const prio_t prio) {

         if(id == 1) {
             Handler12 = prio;
         } else if(id == 2) {
             Handler13 = prio;
         } else if(id == 3) {
             Handler11 = prio;
         } else {
             assert(false);
         }
    }

    /** Set priority of task id to prio **/
    forceinline void increasePrio(const prio_t id, const prio_t prio) {
        
        if(id == 1) {
            if (Handler12 < prio)
                Handler12 = prio;
        } else if(id == 2) {
            if (Handler13 < prio)
                Handler13 = prio;
        } else if(id == 3) {
            if (Handler11 < prio)
                Handler11 = prio;
        } else {
            assert(false);
        }
    }

    forceinline void head(id_t& id, prio_t& prio) const {
        // start with idle id/priority
        id = idle_id;
        prio = idle_prio;



        
        
                if(Handler12 > prio) {
                    prio = Handler12;
                    id = 1;
                }
        
        
                if(Handler13 > prio) {
                    prio = Handler13;
                    id = 2;
                }
        
        
                if(Handler11 > prio) {
                    prio = Handler11;
                    id = 3;
                }
        
        
    }

    forceinline void insert(const id_t& id, const prio_t& prio) {
        increasePrio(id, prio);
    }

    forceinline void remove(const id_t& id) {
        set(id, 0);
    }

    forceinline void promote(const id_t& id, const prio_t& newprio) {
        set(id, newprio);
    }
};

}; // scheduler
}; // os
namespace os {
namespace scheduler {

using namespace os::tasks;

struct Scheduler {
    os::scheduler::TaskList tlist;

    typedef uint8_t prio_t;
    typedef uint8_t id_t;

    prio_t current_prio;
    id_t current_task;

    static constexpr prio_t scheduler_prio = 4;

    forceinline void Reschedule(void) {
        // OPTIMIZATION: do not reschedule if RES_SCHEDULER is taken
        if (current_prio != scheduler_prio) {
            // set current (=next) task from task list
            tlist.head(current_task, current_prio);
        }

        // dispatch or enter idle
        // TODO: generated signature

        if(current_task == OS_Handler12_task.id) {
            if (OS_Handler12_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Handler12_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Handler12_task);
        } else if(current_task == OS_Handler13_task.id) {
            if (OS_Handler13_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Handler13_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Handler13_task);
        } else if(current_task == OS_Handler11_task.id) {
            if (OS_Handler11_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Handler11_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Handler11_task);
        } else if(current_task == TaskList::idle_id) {
            Dispatcher::idle();
        } else {
            assert(false);
        }
    }


    forceinline void SetReady_impl(const Task task) {
        IncreasePriority(task, task.prio);
    }


    forceinline void Schedule_impl(void) {
        if(in_syscall()) {
            // in syscall: reschedule directly
            Reschedule();
        } else {
            // not in syscall (probably in ISR): request reschedule AST
            // Calls also Reschedule()
            request_reschedule_ast();
        }
    }

    forceinline void SetSuspended_impl(const Task t) {
        t.tcb.reset();

        tlist.remove(t.id);
        if (t.preemptable == false) {
            // restore non-preemptable task to original priority
            // this is required for the optimization in Reschedule()
            current_prio = t.prio;
        }
    }

    forceinline void ActivateTask_impl(const Task t) {
        SetReady_impl(t);
        Schedule_impl();
    }

    forceinline void ChainTask_impl(const Task from, const Task to) {
        auto from_id = from.id;
        assert(from_id == current_task);

        SetSuspended_impl(from);
        SetReady_impl(to);
        Schedule_impl();
    }

    forceinline void TerminateTask_impl(const Task from) {
        auto id = from.id;
        assert(id == current_task);

        SetSuspended_impl(from);

        Schedule_impl();
    }

    forceinline void GetResource_impl(const Task current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
    }

    forceinline void ReleaseResource_impl(const Task current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
        Schedule_impl();
    }


    forceinline void SetReadyFromSuspended_impl(const Task task) {
        SetPriority(task, task.prio);
    }

    // Low level interface to the task list
    forceinline void SetCurrentTask(const Task task) {
        current_task = task.id;
    }

    forceinline void SetSystemPriority(const int new_prio) {
        current_prio = new_prio;
    }

    forceinline void SetPriority(const Task task, const int new_prio) {
        tlist.set(task.id, new_prio);
    }

    forceinline void IncreasePriority(const Task task, const int new_prio) {
        

        if (task.id == OS_Handler12_task.id) {
            tlist.insert(OS_Handler12_task.id,  new_prio);
        } else if (task.id == OS_Handler13_task.id) {
            tlist.insert(OS_Handler13_task.id,  new_prio);
        } else if (task.id == OS_Handler11_task.id) {
            tlist.insert(OS_Handler11_task.id,  new_prio);
        } else {
            assert(false);
        }
    }

};

constexpr Scheduler::prio_t Scheduler::scheduler_prio;
Scheduler scheduler_;

noinline void ScheduleC_impl(__attribute__ ((unused)) uint32_t dummy) {
    scheduler_.Reschedule();
}

}; // scheduler
}; // os
#include "counter.h"
#include "alarm.h"

namespace os {

void inlinehint Counter::tick() {
    
    
}

void inlinehint Alarm::checkCounter(const Counter &counter) {
    (void) counter; // unused argument warning might happen

    
    
    
    
    
}


} // os

extern "C" void StartOS(int arg0){
    (void) arg0;
    OS_Handler12_task.tcb.reset();
    OS_Handler13_task.tcb.reset();
    OS_Handler11_task.tcb.reset();
    scheduler_.SetReadyFromSuspended_impl(OS_Handler11_task);
    Machine::enable_interrupts();
    syscall(os::scheduler::ScheduleC_impl, 0);
    Machine::unreachable();
}

extern "C" StatusType OSEKOS_TerminateTask__ABB490(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_Handler13_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Handler11>
         */
        scheduler_.SetCurrentTask(OS_Handler11_task);
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 2 == <Subtask OSEKOS_TASK_Handler11>
         */
        scheduler_.SetSystemPriority(2);
        // OPTIMIZATION: The task is surely ready, just resume it.
        Dispatcher::ResumeToTask(OS_Handler11_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_TerminateTask__ABB486(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_Handler12_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask Idle>
         */
        scheduler_.current_task = TaskList::idle_id;
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 0 == <Subtask Idle>
         */
        scheduler_.SetSystemPriority(0);
        Dispatcher::idle();
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_ActivateTask__ABB456(TaskType arg0){
    // Hook: SystemEnterHook
    {
    }
    (void) arg0;
    Machine::disable_interrupts();
    {
        // OPTIMIZATION: We surely know that the task is suspended
        scheduler_.SetReadyFromSuspended_impl(OS_Handler13_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Handler13>
         */
        scheduler_.SetCurrentTask(OS_Handler13_task);
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 3 == <Subtask OSEKOS_TASK_Handler13>
         */
        scheduler_.SetSystemPriority(3);
        // OPTIMIZATION: The task is surely no paused. We can surely start it from scratch.
        Dispatcher::StartToTask(OS_Handler13_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_ActivateTask__ABB466(TaskType arg0){
    // Hook: SystemEnterHook
    {
    }
    (void) arg0;
    Machine::disable_interrupts();
    {
        // OPTIMIZATION: We surely know that the task is suspended
        scheduler_.SetReadyFromSuspended_impl(OS_Handler12_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Handler11>
         */
        // OPTIMIZATION: We do not have to update the current_task entry
        // OPTIMIZATION: We do not have to update the current system priority
        // OPTIMIZATION: The task is surely ready, just resume it.
        Dispatcher::ResumeToTask(OS_Handler11_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_ActivateTask__ABB472(TaskType arg0){
    // Hook: SystemEnterHook
    {
    }
    (void) arg0;
    Machine::disable_interrupts();
    {
        // OPTIMIZATION: We surely know that the task is suspended
        scheduler_.SetReadyFromSuspended_impl(OS_Handler13_task);
        /* OPTIMIZATION: There is only one possible subtask continuing to, we directly
         * dispatch to that task: <Subtask OSEKOS_TASK_Handler13>
         */
        scheduler_.SetCurrentTask(OS_Handler13_task);
        /* OPTIMIZATION: The system priority is determined. Therefore we set it from
         * a constant: 3 == <Subtask OSEKOS_TASK_Handler13>
         */
        scheduler_.SetSystemPriority(3);
        // OPTIMIZATION: The task is surely no paused. We can surely start it from scratch.
        Dispatcher::StartToTask(OS_Handler13_task);
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

extern "C" StatusType OSEKOS_TerminateTask__ABB479(){
    // Hook: SystemEnterHook
    {
    }
    Machine::disable_interrupts();
    {
        scheduler_.SetSuspended_impl(OS_Handler11_task);
        scheduler_.Reschedule();
    }
    Machine::enable_interrupts();
    // Hook: SystemLeaveHook
    {
    }
    return E_OK;
}

void __OS_HOOK_PreIdleHook(){
    __OS_HOOK_DEFINED_PreIdleHook();
}

