#include "os.h"
#include "tcb.h"
#include "os/scheduler/task.h"
#include "os/scheduler/tasklist.h"
#include "os/scheduler/scheduler.h"
#include "os/alarm.h"
#include "irq.h"

extern "C" void  OSEKOS_TASK_Task1();
extern "C" void  OSEKOS_TASK_Task2();
extern "C" void  __OS_StartOS_dispatch(int);
extern "C" void  StartOS(int);
extern "C" void  __OS_ASTSchedule(int);
extern "C" void  OSEKOS_kickoff__ABB442();
extern "C" void  OSEKOS_kickoff__ABB419();
extern "C" void  OSEKOS_kickoff__ABB410();
extern "C" void  OSEKOS_kickoff__ABB454();
extern "C" StatusType  OSEKOS_TerminateTask__ABB455();
extern "C" void  OSEKOS_kickoff__ABB618();
extern "C" StatusType  OSEKOS_TerminateTask__ABB619();
extern "C" void  OSEKOS_kickoff__ABB451();
extern "C" StatusType  OSEKOS_ActivateTask__ABB452(TaskType);
extern "C" void  OSEKOS_kickoff__ABB1119();
void  __OS_HOOK_PreIdleHook();
extern "C" void  OSEKOS_ISR_ISR37(int);
extern "C" void  OSEKOS_ISR_ISR38(int);
extern "C" void  OSEKOS_ISR_ISR39(int);
extern "C" void  OSEKOS_ISR_ISR40(int);

namespace os{
    namespace tasks{
        extern const os::scheduler::Task OS_Task1_task;
        extern const os::scheduler::Task OS_Task2_task;
    }
}

extern "C" uint8_t Task1_stack[4096];
extern "C" uint8_t Task2_stack[4096];
namespace arch{
    extern void * OS_Task1_stackptr;
    extern void * OS_Task2_stackptr;
    extern const arch::TCB OS_Task1_tcb;
    extern const arch::TCB OS_Task2_tcb;
}


namespace os{
    namespace tasks{
        constexpr const os::scheduler::Task OS_Task1_task(1, 1, true, arch::OS_Task1_tcb);
        constexpr const os::scheduler::Task OS_Task2_task(2, 2, true, arch::OS_Task2_tcb);
    }
}

uint8_t Task1_stack[4096];
uint8_t Task2_stack[4096];
namespace arch{
    void * OS_Task1_stackptr;
    void * OS_Task2_stackptr;
    constexpr const arch::TCB OS_Task1_tcb(&OSEKOS_TASK_Task1, Task1_stack, OS_Task1_stackptr, 4096);
    constexpr const arch::TCB OS_Task2_tcb(&OSEKOS_TASK_Task2, Task2_stack, OS_Task2_stackptr, 4096);
}

using namespace os::tasks;
using namespace arch;

// ****************************************************************
// TaskList Implementation
// ****************************************************************

namespace os { namespace scheduler {

// only needed for Tasklist constructor comma placement
class UnencodedTaskListStatic {};

/* Simpler array based task queue */
struct TaskList : public UnencodedTaskListStatic {
    typedef uint8_t id_t;
    typedef uint8_t prio_t;

    // encoded task priorities
    
    prio_t Task1;
    prio_t Task2;
    

    // idle task id/priority
    static constexpr id_t idle_id = 0;
    static constexpr prio_t idle_prio = 0;

    
    TaskList() : UnencodedTaskListStatic() , Task1(0), Task2(0) {}

    forceinline bool isSuspended(const prio_t id) {
    
        if(id == 1) {
            return Task1 == 0;
        } else if(id == 2) {
            return Task2 == 0;
        } else {
            assert(false);
            return false;
        }
    }

    /** Set priority of task id to prio **/
    forceinline void set(const prio_t id, const prio_t prio) {

         if(id == 1) {
             Task1 = prio;
         } else if(id == 2) {
             Task2 = prio;
         } else {
             assert(false);
         }
    }



    /** Set priority of task id to prio **/
    forceinline void increasePrio(const prio_t id, const prio_t prio) {
        
        if(id == 1) {
            if (Task1 < prio)
                Task1 = prio;
        } else if(id == 2) {
            if (Task2 < prio)
                Task2 = prio;
        } else {
            assert(false);
        }
    }

    template<typename TargetHint>
    forceinline void head(id_t& id, prio_t& prio) const {
        // start with idle id/priority
        id = idle_id;
        prio = idle_prio;



        
        
                if(TargetHint::Task1_is_possible && Task1 > prio) {
                    prio = Task1;
                    id = 1;
                }
        
        
                if(TargetHint::Task2_is_possible && Task2 > prio) {
                    prio = Task2;
                    id = 2;
                }
        
        
    }

    forceinline void insert(const id_t& id, const prio_t& prio) {
        increasePrio(id, prio);
    }

    forceinline void remove(const id_t& id) {
        set(id, 0);
    }

    forceinline void promote(const id_t& id, const prio_t& newprio) {
        set(id, newprio);
    }
};

}; // scheduler
}; // os
namespace os {
namespace scheduler {

using namespace os::tasks;

 
 

template<bool Idle = true, bool Task1 = true, bool Task2 = true>
struct SchedulerTargetHint {
    static const bool Idle_is_possible = Idle;
    static const bool Task1_is_possible = Task1;
             static const bool Task2_is_possible = Task2;
             
};

struct Scheduler {
    os::scheduler::TaskList tlist;

    typedef uint8_t prio_t;
    typedef uint8_t id_t;

    prio_t current_prio;
    id_t current_task;

    static constexpr prio_t scheduler_prio = 3;

    template<typename TargetHint = SchedulerTargetHint<> >
    forceinline void Reschedule(void) {
        // OPTIMIZATION: do not reschedule if RES_SCHEDULER is taken
        if (current_prio != scheduler_prio) {
            // set current (=next) task from task list
            tlist.head<TargetHint>(current_task, current_prio);
        }

        // dispatch or enter idle
        // TODO: generated signature

        if(TargetHint::Task1_is_possible && current_task == OS_Task1_task.id) {
            if (OS_Task1_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Task1_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Task1_task);
        } else if(TargetHint::Task2_is_possible && current_task == OS_Task2_task.id) {
            if (OS_Task2_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Task2_task.id, scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Task2_task);
        } else if(TargetHint::Idle_is_possible && current_task == TaskList::idle_id) {
            Dispatcher::idle();
        } else {
            assert(false);
        }
    }


    forceinline void SetReady_impl(const Task task) {
        IncreasePriority(task, task.prio);
    }


    forceinline void Schedule_impl(void) {
        if(in_syscall()) {
            // in syscall: reschedule directly
            Reschedule();
        } else {
            // not in syscall (probably in ISR): request reschedule AST
            // Calls also Reschedule()
            request_reschedule_ast();
        }
    }

    forceinline void SetSuspended_impl(const Task t) {
        t.tcb.reset();

        tlist.remove(t.id);
        if (t.preemptable == false) {
            // restore non-preemptable task to original priority
            // this is required for the optimization in Reschedule()
            current_prio = t.prio;
        }
    }

    forceinline void ActivateTask_impl(const Task t) {
        SetReady_impl(t);
        Schedule_impl();
    }

    forceinline void ChainTask_impl(const Task from, const Task to) {
        auto from_id = from.id;
        assert(from_id == current_task);

        SetSuspended_impl(from);
        SetReady_impl(to);
        Schedule_impl();
    }

    forceinline void TerminateTask_impl(const Task from) {
        auto id = from.id;
        assert(id == current_task);

        SetSuspended_impl(from);

        Schedule_impl();
    }

    forceinline void GetResource_impl(const Task current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
    }

    forceinline void ReleaseResource_impl(const Task current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
        Schedule_impl();
    }


    forceinline void SetReadyFromSuspended_impl(const Task task) {
        SetPriority(task, task.prio);
    }

    // Low level interface to the task list
    forceinline void SetCurrentTask(const Task task) {
        if (task.preemptable == false) {
            // promote non-preemptable task to RES_SCHEDULER
            tlist.set(task.id, scheduler_prio);
            current_prio = scheduler_prio;
        }
        current_task = task.id;
    }

    forceinline void SetSystemPriority(const int new_prio) {
        current_prio = new_prio;
    }

    forceinline void SetPriority(const Task task, const int new_prio) {
        tlist.set(task.id, new_prio);
    }

    forceinline bool isSuspended(const Task task) {
        return tlist.isSuspended(task.id);
    }

    forceinline void IncreasePriority(const Task task, const int new_prio) {
        

        if (task.id == OS_Task1_task.id) {
            tlist.insert(OS_Task1_task.id,  new_prio);
        } else if (task.id == OS_Task2_task.id) {
            tlist.insert(OS_Task2_task.id,  new_prio);
        } else {
            assert(false);
        }
    }

};

constexpr Scheduler::prio_t Scheduler::scheduler_prio;
Scheduler scheduler_;

}; // scheduler
}; // os
#include "counter.h"
#include "alarm.h"

namespace os {

void inlinehint Counter::tick() {
    
    
}

void inlinehint Alarm::checkCounter(const Counter &counter) {
    (void) counter; // unused argument warning might happen

    
    
    
    
}


} // os

extern "C" void __OS_StartOS_dispatch(int arg0){
    (void) arg0;
    scheduler_.Reschedule();
}

extern "C" void StartOS(int arg0){
    arch::irq.set_handler(40, OSEKOS_ISR_ISR40);
    arch::irq.enable(40);
    arch::irq.set_handler(39, OSEKOS_ISR_ISR39);
    arch::irq.enable(39);
    arch::irq.set_handler(38, OSEKOS_ISR_ISR38);
    arch::irq.enable(38);
    arch::irq.set_handler(37, OSEKOS_ISR_ISR37);
    arch::irq.enable(37);
    (void) arg0;
    OS_Task1_task.tcb.reset();
    scheduler_.SetReadyFromSuspended_impl(OS_Task1_task);
    OS_Task2_task.tcb.reset();
    syscall(__OS_StartOS_dispatch);
    Machine::unreachable();
}

extern "C" void __OS_ASTSchedule(int arg0){
    (void) arg0;
    scheduler_.Reschedule();
}

extern "C" void OSEKOS_kickoff__ABB442(){
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
}

extern "C" void OSEKOS_kickoff__ABB419(){
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
}

extern "C" void OSEKOS_kickoff__ABB410(){
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
}

extern "C" void OSEKOS_kickoff__ABB454(){
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
}

extern "C" StatusType OSEKOS_TerminateTask__ABB455(){
    Machine::disable_interrupts();
    // Hook: SystemEnterHook
    {
    }
    {
        scheduler_.TerminateTask_impl(OS_Task1_task);
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    return E_OK;
}

extern "C" void OSEKOS_kickoff__ABB618(){
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
}

extern "C" StatusType OSEKOS_TerminateTask__ABB619(){
    Machine::disable_interrupts();
    // Hook: SystemEnterHook
    {
    }
    {
        scheduler_.TerminateTask_impl(OS_Task2_task);
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    return E_OK;
}

extern "C" void OSEKOS_kickoff__ABB451(){
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
}

extern "C" StatusType OSEKOS_ActivateTask__ABB452(TaskType arg0){
    // Called from ISR, no disable interrupts required!
    // Hook: SystemEnterHook
    {
    }
    {
        scheduler_.ActivateTask_impl(OS_Task2_task);
    }
    // Hook: SystemLeaveHook
    {
    }
    // Called from ISR, no enable interrupts required!
    (void) arg0;
    return E_OK;
}

extern "C" void OSEKOS_kickoff__ABB1119(){
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
}

void __OS_HOOK_PreIdleHook(){
    OSEKOS_kickoff__ABB1119();
}

