#include "os.h"
#include "tcb.h"
#include "os/scheduler/task.h"
#include "os/scheduler/tasklist.h"
#include "os/scheduler/scheduler.h"
#include "os/alarm.h"

extern "C" void  OSEKOS_TASK_Handler11();
extern "C" void  OSEKOS_TASK_Handler12();
extern "C" void  OSEKOS_TASK_Handler13();
extern "C" void  __OS_StartOS_dispatch(int);
extern "C" void  StartOS(int);
extern "C" void  __OS_ASTSchedule(int);
extern "C" void inlinehint OSEKOS_kickoff__ABB38();
extern "C" void  __OS_syscall_OSEKOS_ActivateTask__ABB41();
extern "C" StatusType inlinehint OSEKOS_ActivateTask__ABB41(TaskType);
extern "C" void  __OS_syscall_OSEKOS_ActivateTask__ABB45();
extern "C" StatusType inlinehint OSEKOS_ActivateTask__ABB45(TaskType);
extern "C" void  __OS_syscall_OSEKOS_TerminateTask__ABB49();
extern "C" StatusType inlinehint OSEKOS_TerminateTask__ABB49();
extern "C" void inlinehint OSEKOS_kickoff__ABB232();
extern "C" void  __OS_syscall_OSEKOS_TerminateTask__ABB235();
extern "C" StatusType inlinehint OSEKOS_TerminateTask__ABB235();
extern "C" void inlinehint OSEKOS_kickoff__ABB535();
extern "C" void  __OS_syscall_OSEKOS_TerminateTask__ABB538();
extern "C" StatusType inlinehint OSEKOS_TerminateTask__ABB538();
extern "C" void inlinehint OSEKOS_kickoff__ABB552();
void  __OS_HOOK_PreIdleHook();

namespace os{
    namespace tasks{
        extern const os::scheduler::Task OS_Handler11_task;
        extern const os::scheduler::Task OS_Handler12_task;
        extern const os::scheduler::Task OS_Handler13_task;
    }
}

namespace arch{
    extern void * OS_Handler11_stackptr;
    extern void * OS_Handler12_stackptr;
    extern void * OS_Handler13_stackptr;
    extern void * const OS_stackptrs[];
    extern const arch::TCB OS_Handler11_tcb;
    extern const arch::TCB OS_Handler12_tcb;
    extern const arch::TCB OS_Handler13_tcb;
    extern const TCB * const OS_tcbs[];
}

extern "C" uint8_t Handler11_stack[4096];
extern "C" uint8_t Handler12_stack[4096];
extern "C" uint8_t Handler13_stack[4096];

namespace os{
    namespace tasks{
        constexpr const os::scheduler::Task OS_Handler11_task(1, 2, true, arch::OS_Handler11_tcb);
        constexpr const os::scheduler::Task OS_Handler12_task(2, 1, true, arch::OS_Handler12_tcb);
        constexpr const os::scheduler::Task OS_Handler13_task(3, 3, true, arch::OS_Handler13_tcb);
    }
}

namespace arch{
    void * OS_Handler11_stackptr;
    void * OS_Handler12_stackptr;
    void * OS_Handler13_stackptr;
    void * const OS_stackptrs[] = {/* 0 */ &startup_sp, /* 1 */ &OS_Handler11_stackptr, /* 2 */ &OS_Handler12_stackptr, /* 3 */ &OS_Handler13_stackptr};
    constexpr const arch::TCB OS_Handler11_tcb(&OSEKOS_TASK_Handler11, Handler11_stack, OS_Handler11_stackptr, 4096);
    constexpr const arch::TCB OS_Handler12_tcb(&OSEKOS_TASK_Handler12, Handler12_stack, OS_Handler12_stackptr, 4096);
    constexpr const arch::TCB OS_Handler13_tcb(&OSEKOS_TASK_Handler13, Handler13_stack, OS_Handler13_stackptr, 4096);
    const TCB * const OS_tcbs[] = {/* 0 */ 0, /* 1 */ &OS_Handler11_tcb, /* 2 */ &OS_Handler12_tcb, /* 3 */ &OS_Handler13_tcb};
}

uint8_t Handler11_stack[4096];
uint8_t Handler12_stack[4096];
uint8_t Handler13_stack[4096];
using namespace os::tasks;
using namespace arch;

// ****************************************************************
// TaskList Implementation
// ****************************************************************

namespace os { namespace scheduler {

/* Simpler array based task queue */
struct TaskList : public TaskListStatic {
    // encoded task priorities
    
    Encoded_Static<A0, 1004> Handler11;
    Encoded_Static<A0, 1006> Handler12;
    Encoded_Static<A0, 1008> Handler13;
    

    // idle task id/priority
    static constexpr auto idle_id   = EC(1009, 0);
    static constexpr auto idle_prio = EC(1010, 0);

    
    TaskList() : TaskListStatic(), Handler11(0), Handler12(0), Handler13(0) {}

    /** Set priority of task id to prio **/
    // returns an encoded 0 with the signature (B) of the modified task - prio.B
    template<typename T>
    forceinline value_coded_t isSuspended(const T id) {
        

        if(id == 1) {
            return Handler11.getCodedValue() == EC(decltype(Handler11)::B, 0).getCodedValue();
        } else if(id == 2) {
            return Handler12.getCodedValue() == EC(decltype(Handler12)::B, 0).getCodedValue();
        } else if(id == 3) {
            return Handler13.getCodedValue() == EC(decltype(Handler13)::B, 0).getCodedValue();
        } else {
            assert(false);
            return 0;
        }
    }

    /** Set priority of task id to prio **/
    // returns an encoded 0 with the signature (B) of the modified task - prio.B
    template<typename T, typename S>
    forceinline value_coded_t set(const T id, const S prio) {


         if(id == 1) {
             Handler11 = prio;
             return (Handler11 - prio).getCodedValue();
         } else if(id == 2) {
             Handler12 = prio;
             return (Handler12 - prio).getCodedValue();
         } else if(id == 3) {
             Handler13 = prio;
             return (Handler13 - prio).getCodedValue();
         } else {
             assert(false);
             return 0;
         }
    }

    /** Set priority of task id to prio **/
    // returns an encoded 0 with the signature (B) of the modified task - prio.B
    template<typename T, typename S>
    forceinline value_coded_t increasePrio(const T id, const S newprio) {
        

        if(id == 1) {
            Handler11.vc += 10;
            updateMax<10, 11>(Handler11, newprio);
            Handler11.vc -= 11;
            value_coded_t ret = (Handler11 - newprio).getCodedValue();
            assert (ret >= 0);
            return ret;
        } else if(id == 2) {
            Handler12.vc += 10;
            updateMax<10, 11>(Handler12, newprio);
            Handler12.vc -= 11;
            value_coded_t ret = (Handler12 - newprio).getCodedValue();
            assert (ret >= 0);
            return ret;
        } else if(id == 3) {
            Handler13.vc += 10;
            updateMax<10, 11>(Handler13, newprio);
            Handler13.vc -= 11;
            value_coded_t ret = (Handler13 - newprio).getCodedValue();
            assert (ret >= 0);
            return ret;
        } else {
            assert(false);
            return 0;
        }
    }

    template<typename TargetHint, typename T, typename S>
    forceinline value_coded_t head(T& id, S& prio) const {
        B_t sigId = id.getB();
        B_t sigPrio = prio.getB();

        bool first = true;



        
        // This branch is taken statically
        if (TargetHint::Handler11_is_possible) {
            if(first) {
                first=false;
                sigId += 17;
                sigPrio += 17;
                id.vc = Encoded::encode(1, A0, sigId, 0);
                prio.vc = Handler11.vc + (sigPrio - Handler11.getB());
            } else {
                // Handler11 >= prio?
                updateMax(sigPrio, sigId, 17, &prio, Handler11, &id, EC(1003, 1));
        
                assert((prio.vc + id.vc) % A0 == (sigPrio + sigId));
            }
        }
        
        // This branch is taken statically
        if (TargetHint::Handler12_is_possible) {
            if(first) {
                first=false;
                sigId += 18;
                sigPrio += 18;
                id.vc = Encoded::encode(2, A0, sigId, 0);
                prio.vc = Handler12.vc + (sigPrio - Handler12.getB());
            } else {
                // Handler12 >= prio?
                updateMax(sigPrio, sigId, 18, &prio, Handler12, &id, EC(1005, 2));
        
                assert((prio.vc + id.vc) % A0 == (sigPrio + sigId));
            }
        }
        
        // This branch is taken statically
        if (TargetHint::Handler13_is_possible) {
            if(first) {
                first=false;
                sigId += 19;
                sigPrio += 19;
                id.vc = Encoded::encode(3, A0, sigId, 0);
                prio.vc = Handler13.vc + (sigPrio - Handler13.getB());
            } else {
                // Handler13 >= prio?
                updateMax(sigPrio, sigId, 19, &prio, Handler13, &id, EC(1007, 3));
        
                assert((prio.vc + id.vc) % A0 == (sigPrio + sigId));
            }
        }
        
        if (TargetHint::Idle_is_possible) {
            if(first) {
                first=false;
                id=idle_id;
                prio=idle_prio;
            } else {
                // restore idle_id if idle_id >= prio
                updateMax(sigPrio, sigId, 1014, &prio, idle_prio, &id, idle_id);
        
                // last comparison, needs no assert
            }
        }
        

        pseudo_static_assert(!first, "no task possible");

        pseudo_static_assert(sigId > 0, "constant sigId not optimized away completely");
        pseudo_static_assert(sigPrio > 0, "constant sigPrio not optimized away completely");

        id.vc -= (sigId - id.getB());
        prio.vc -= (sigPrio - prio.getB());

        return 0;
    }

    template<typename TID, typename NewPrio>
        forceinline value_coded_t insert(const TID& id, const NewPrio &newprio) {

        return increasePrio(id, newprio);
    }

    template<typename T>
    forceinline value_coded_t remove(const T& id) {
        // The signature here is arbitrary, and won't be exposed.
        return set(id, EC(3, 0));
    }

    template<typename T, typename S>
    forceinline value_coded_t promote(const T& id, const S& newprio) {

        return set(id, newprio);
    }

    void dump() {
        
        debug << "Handler11: " << Handler11.decode() << endl;
        debug << "Handler12: " << Handler12.decode() << endl;
        debug << "Handler13: " << Handler13.decode() << endl;
        
    }
    /* unused by scheduler
    template<typename T, typename S>
    forceinline value_coded_t dequeue(T& id, S& prio) {
        static value_coded_t sig1;

        sig1 = head(id, prio);

        value_coded_t sig2;
        if(prio != idle_prio) {
            sig2 = remove(id);
            // IDEA: set id.vc += sig2 + X; ?
        } else {
            // IDEA: set id.vc = sig2 + X; ?
            // The 42 here is choosen by fair dice roll
            sig2 = 42;
        }

        // TODO: more control flow checks?

        return sig1+sig2;
    }
    */
};

constexpr Encoded_Static<A0, 1009> TaskList::idle_id;
constexpr Encoded_Static<A0, 1010> TaskList::idle_prio;

}; // scheduler
}; // os
namespace os {
namespace scheduler {

using namespace os::tasks;




template<bool Idle = true, bool Handler11 = true, bool Handler12 = true, bool Handler13 = true>
struct SchedulerTargetHint {
    static const bool Idle_is_possible = Idle;
    static const bool Handler11_is_possible = Handler11;
    static const bool Handler12_is_possible = Handler12;
    static const bool Handler13_is_possible = Handler13;
    
};

struct Scheduler  {
    os::scheduler::TaskList tlist;

    Encoded_Static<A0, 99> current_prio;
    Encoded_Static<A0, 98> current_task;

    static constexpr auto scheduler_prio = EC(1001, 4);

    template<typename TargetHint = SchedulerTargetHint<> >
    forceinline void Reschedule(void) {
        // TODO: control flow check
        //  tlist.dump();

        // OPTIMIZATION: do not reschedule if RES_SCHEDULER is taken
        if (current_prio != scheduler_prio) {
            // set current (=next) task from task list
            tlist.head<TargetHint>(current_task, current_prio);
        }
    assert(current_task.check());
    assert(current_prio.check());

        // dispatch or enter idle
        // TODO: generated signature

        if(TargetHint::Handler11_is_possible && current_task == OS_Handler11_task.enc_id<1>()) {
            if (OS_Handler11_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Handler11_task.enc_id<1>(), scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Handler11_task);
        } else if(TargetHint::Handler12_is_possible && current_task == OS_Handler12_task.enc_id<1>()) {
            if (OS_Handler12_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Handler12_task.enc_id<1>(), scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Handler12_task);
        } else if(TargetHint::Handler13_is_possible && current_task == OS_Handler13_task.enc_id<1>()) {
            if (OS_Handler13_task.preemptable == false) {
                // promote non-preemptable task to RES_SCHEDULER
                tlist.promote(OS_Handler13_task.enc_id<1>(), scheduler_prio);
                current_prio = scheduler_prio;
            }
            Dispatcher::Dispatch(OS_Handler13_task);
        } else if(TargetHint::Idle_is_possible && current_task == TaskList::idle_id) {
            Dispatcher::idle();
        } else {
            assert(false);
        }
    }

    template<int TASK_SIG = 1015>
    forceinline void SetReady_impl(const Task &task) {
        IncreasePriority<TASK_SIG>(task, task.prio);
    }

    forceinline void Schedule_impl(void) {
        if(in_syscall()) {
            // in syscall: reschedule directly
            Reschedule();
        } else {
            // not in syscall (probably in ISR): request reschedule AST
            // Calls also Reschedule()
            request_reschedule_ast();
        }
    }

    template<int TASK_SIG = 1016>
    forceinline void SetSuspended_impl(const Task &t) {
        t.tcb.reset();

        tlist.remove(t.enc_id<TASK_SIG>());
        if (t.preemptable == false) {
            // restore non-preemptable task to original priority
            // this is required for the optimization in Reschedule()
            current_prio = t.enc_prio<1017>();
        }
    }

    template<int TO_SIG = 1018>
    forceinline void ActivateTask_impl(const Task &to) {
        SetReady_impl<TO_SIG>(to);
        Schedule_impl();
    }

    template<int FROM_SIG, int TO_SIG>
    forceinline void ChainTask_impl(const Task &from, const Task &to) {
        auto from_id = from.enc_id<FROM_SIG>();
        assert(from_id == current_task);

        SetSuspended_impl<FROM_SIG>(from);
        SetReady_impl<TO_SIG>(to);
        Schedule_impl();
    }

    forceinline void TerminateTask_impl(const Task &from) {
        auto id = from.enc_id<1019>();
        assert(id == current_task);

        SetSuspended_impl(from);

        Schedule_impl();
    }

    forceinline void GetResource_impl(const Task &current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
    }

    forceinline void ReleaseResource_impl(const Task &current_task, const int new_prio) {
        SetPriority(current_task, new_prio);
        SetSystemPriority(new_prio);
        Schedule_impl();
    }

    forceinline void SetReadyFromSuspended_impl(const Task &task) {
        SetPriority(task, task.prio);
    }

    // Low level interface to the task list
    template<int TASK_SIG = 1020>
    forceinline void SetCurrentTask(const Task &task) {
        if (task.preemptable == false) {
            // promote non-preemptable task to RES_SCHEDULER
            tlist.set(task.enc_id<1>(), scheduler_prio);
            current_prio = scheduler_prio;
        }
        current_task = task.enc_id<TASK_SIG>();
    }

    forceinline void SetSystemPriority(const int new_prio) {
        auto new_prio_encoded = EC(2, new_prio);
        current_prio = new_prio_encoded;
    }


    forceinline void SetPriority(const Task &task, const int new_prio) {
        auto id = task.enc_id<1>();
        auto new_prio_encoded = EC(2, new_prio);
        tlist.set(id, new_prio_encoded);
    }

    forceinline bool isSuspended(const Task &task) {
        auto id = task.enc_id<1>();
        return tlist.isSuspended(id);
    }


    template<int TASK_SIG = 1021>
    forceinline void IncreasePriority(const Task &task, const int new_prio) {
    auto id = task.enc_id<TASK_SIG>();
        // TODO: generated signature
        
        if (id == OS_Handler11_task.enc_id<1>()) {
            value_coded_t signature = tlist.insert(OS_Handler11_task.enc_id<1000>(), EC(999, new_prio));
            assert((signature - (1004 - 999)) % decltype(id)::A == 0);
        } else if (id == OS_Handler12_task.enc_id<1>()) {
            value_coded_t signature = tlist.insert(OS_Handler12_task.enc_id<998>(), EC(997, new_prio));
            assert((signature - (1006 - 997)) % decltype(id)::A == 0);
        } else if (id == OS_Handler13_task.enc_id<1>()) {
            value_coded_t signature = tlist.insert(OS_Handler13_task.enc_id<996>(), EC(995, new_prio));
            assert((signature - (1008 - 995)) % decltype(id)::A == 0);
        } else {
            assert(false);
        }
    }
};

constexpr Encoded_Static<A0, 1001> Scheduler::scheduler_prio;
Scheduler scheduler_;


}; // namespace scheduler
}; // namespace os
#include "counter.h"
#include "alarm.h"

namespace os {

void inlinehint Counter::tick() {
    
    
}

void inlinehint Alarm::checkCounter(const Counter &counter) {
    (void) counter; // unused argument warning might happen

    
    
    
    
}


} // os

extern "C" void __OS_StartOS_dispatch(int arg0){
    (void) arg0;
    scheduler_.Reschedule();
}

extern "C" void StartOS(int arg0){
    (void) arg0;
    OS_Handler11_task.tcb.reset();
    scheduler_.SetReadyFromSuspended_impl(OS_Handler11_task);
    OS_Handler12_task.tcb.reset();
    OS_Handler13_task.tcb.reset();
    syscall(__OS_StartOS_dispatch);
    Machine::unreachable();
}

extern "C" void __OS_ASTSchedule(int arg0){
    (void) arg0;
    scheduler_.Reschedule();
}

extern "C" void OSEKOS_kickoff__ABB38(){
    asm_label("syscall_start_OSEKOS_kickoff__ABB38");
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_kickoff__ABB38");
}

extern "C" void __OS_syscall_OSEKOS_ActivateTask__ABB41(){
    // Hook: SystemEnterHook
    {
    }
    scheduler_.ActivateTask_impl<1022>(OS_Handler12_task);
}

extern "C" StatusType OSEKOS_ActivateTask__ABB41(TaskType arg0){
    asm_label("syscall_start_OSEKOS_ActivateTask__ABB41");
    asm_label("syscall_start_OSEKOS_ActivateTask__ABB41");
    syscall(__OS_syscall_OSEKOS_ActivateTask__ABB41);
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_ActivateTask__ABB41");
    (void) arg0;
    asm_label("syscall_end_OSEKOS_ActivateTask__ABB41");
    return E_OK;
}

extern "C" void __OS_syscall_OSEKOS_ActivateTask__ABB45(){
    // Hook: SystemEnterHook
    {
    }
    scheduler_.ActivateTask_impl<1023>(OS_Handler13_task);
}

extern "C" StatusType OSEKOS_ActivateTask__ABB45(TaskType arg0){
    asm_label("syscall_start_OSEKOS_ActivateTask__ABB45");
    asm_label("syscall_start_OSEKOS_ActivateTask__ABB45");
    syscall(__OS_syscall_OSEKOS_ActivateTask__ABB45);
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_ActivateTask__ABB45");
    (void) arg0;
    asm_label("syscall_end_OSEKOS_ActivateTask__ABB45");
    return E_OK;
}

extern "C" void __OS_syscall_OSEKOS_TerminateTask__ABB49(){
    // Hook: SystemEnterHook
    {
    }
    scheduler_.TerminateTask_impl(OS_Handler11_task);
}

extern "C" StatusType OSEKOS_TerminateTask__ABB49(){
    asm_label("syscall_start_OSEKOS_TerminateTask__ABB49");
    asm_label("syscall_start_OSEKOS_TerminateTask__ABB49");
    syscall(__OS_syscall_OSEKOS_TerminateTask__ABB49);
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_TerminateTask__ABB49");
    asm_label("syscall_end_OSEKOS_TerminateTask__ABB49");
    return E_OK;
}

extern "C" void OSEKOS_kickoff__ABB232(){
    asm_label("syscall_start_OSEKOS_kickoff__ABB232");
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_kickoff__ABB232");
}

extern "C" void __OS_syscall_OSEKOS_TerminateTask__ABB235(){
    // Hook: SystemEnterHook
    {
    }
    scheduler_.TerminateTask_impl(OS_Handler13_task);
}

extern "C" StatusType OSEKOS_TerminateTask__ABB235(){
    asm_label("syscall_start_OSEKOS_TerminateTask__ABB235");
    asm_label("syscall_start_OSEKOS_TerminateTask__ABB235");
    syscall(__OS_syscall_OSEKOS_TerminateTask__ABB235);
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_TerminateTask__ABB235");
    asm_label("syscall_end_OSEKOS_TerminateTask__ABB235");
    return E_OK;
}

extern "C" void OSEKOS_kickoff__ABB535(){
    asm_label("syscall_start_OSEKOS_kickoff__ABB535");
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_kickoff__ABB535");
}

extern "C" void __OS_syscall_OSEKOS_TerminateTask__ABB538(){
    // Hook: SystemEnterHook
    {
    }
    scheduler_.TerminateTask_impl(OS_Handler12_task);
}

extern "C" StatusType OSEKOS_TerminateTask__ABB538(){
    asm_label("syscall_start_OSEKOS_TerminateTask__ABB538");
    asm_label("syscall_start_OSEKOS_TerminateTask__ABB538");
    syscall(__OS_syscall_OSEKOS_TerminateTask__ABB538);
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_TerminateTask__ABB538");
    asm_label("syscall_end_OSEKOS_TerminateTask__ABB538");
    return E_OK;
}

extern "C" void OSEKOS_kickoff__ABB552(){
    asm_label("syscall_start_OSEKOS_kickoff__ABB552");
    // Hook: SystemEnterHook
    {
    }
    // Hook: SystemLeaveHook
    {
    }
    Machine::enable_interrupts();
    asm_label("syscall_end_OSEKOS_kickoff__ABB552");
}

void __OS_HOOK_PreIdleHook(){
    OSEKOS_kickoff__ABB552();
    __OS_HOOK_DEFINED_PreIdleHook();
}

